<?php

/**
 * WooCommerce B2B Frontend Checkout set-up Class
 *
 * @version 3.1.4
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_Frontend_Checkout Class
 */
class WCB2B_Frontend_Checkout {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Init current class hooks
     */
    public function init_hooks() {
        add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'add_total_weight' ) );
        add_filter( 'woocommerce_available_payment_gateways', array( $this, 'disable_payment_methods' ) );
        add_filter( 'woocommerce_package_rates', array( $this, 'disable_shipping_methods' ) );
        add_action( 'woocommerce_cart_calculate_fees', array( $this, 'packaging_fee' ) );
        add_filter( 'woocommerce_coupon_is_valid', array( $this, 'coupon_is_valid' ), 10, 3 );
        add_action( 'wp_loaded', array( $this, 'process_quick_order' ) );
        add_action( 'woocommerce_thankyou', array( $this, 'display_purchaseorder_number' ), 1 );
        add_action( 'woocommerce_checkout_process', array( $this, 'check_unpaid_order_amount_limit' ) );

        if ( get_option( 'wcb2b_add_invoice_email' ) === 'yes' ) {
            add_filter( 'woocommerce_billing_fields', array( $this, 'invoice_email_add_field' ) );
            add_action( 'woocommerce_after_checkout_validation', array( $this, 'invoice_email_validation_checkout' ), 10, 2 );
        }

        if ( get_option( 'wcb2b_add_vatnumber' ) === 'yes' ) {
            add_filter( 'woocommerce_billing_fields', array( $this, 'vatnumber_add_field' ) );
            add_action( 'woocommerce_after_checkout_validation', array( $this, 'vatnumber_validation_checkout' ), 10, 2 );
        }

        if ( get_option( 'wcb2b_min_purchase_amount' ) === 'yes' ) {
            add_action( 'woocommerce_after_calculate_totals', array( $this, 'disable_cart_button' ) );
            add_action( 'woocommerce_after_calculate_totals', array( $this, 'disable_checkout_button' ) );
        }
    }

    /**
     * Add total weight to order
     *
     * @param int $order_id Current order ID
     */
    public function add_total_weight( $order_id ) {
        $weight = WC()->cart->get_cart_contents_weight();
        update_post_meta( $order_id, '_total_weight', $weight );
    }

    /**
     * Remove not allowed payment gateways
     * 
     * @param array $available_gateways List of available payment methods
     * @return array
     */
    public function disable_payment_methods( $available_gateways ) {
        if ( is_user_logged_in() && current_user_can( 'manage_woocommerce' ) ) { return $available_gateways; }

        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        // Retrieve group allowed gateways value
        $group_gateways = get_post_meta( $customer_group_id, 'wcb2b_group_gateways', true );
        if ( is_array( $group_gateways ) ) {
            foreach ( $available_gateways as $available_gateway => $data ) {
                // Disable payment method
                if ( in_array( $available_gateway, $group_gateways ) ) {
                    unset( $available_gateways[$available_gateway] );
                }
            }
        }
        return $available_gateways;
    }

    /**
     * Remove not allowed shipping methods
     * 
     * @param array $rates List of available rates
     * @return array
     */
    public function disable_shipping_methods( $shipping_rates ) {
        if ( is_user_logged_in() && current_user_can( 'manage_woocommerce' ) ) { return $shipping_rates; }

        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }
        
        // Retrieve group allowed gateways value
        $group_shippings = get_post_meta( $customer_group_id, 'wcb2b_group_shippings', true );
        if ( is_array( $group_shippings ) ) {
            foreach ( $shipping_rates as $shipping_rate_key => $shipping_rate ) {
                if ( in_array( $shipping_rate_key, $group_shippings ) ) {
                    unset( $shipping_rates[$shipping_rate_key] );
                }
                if ( in_array( $shipping_rate->get_method_id(), $group_shippings ) ) {
                    unset( $shipping_rates[$shipping_rate_key] );
                }
            }
        }
        return $shipping_rates;
    }

    /**
     * Add group package fee to order
     */
    public function packaging_fee() {
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        $packaging_fee = get_post_meta( $customer_group_id, 'wcb2b_group_packaging_fee', true );
        if ( ! empty( $packaging_fee ) ) {
            $value = $packaging_fee['value'];
            if ( $value ) {
                if ( $packaging_fee['type'] == 'percent' ) {
                    $value = floatval( WC()->cart->get_cart_contents_total() ) * $packaging_fee['value'] / 100;
                }
                WC()->cart->add_fee( __( 'Packaging fee', 'woocommerce-b2b' ), $value );
            }
        }
    }

    /**
     * Check coupons restriction by group or by total spent
     *
     * @param boolean $is_valid Check if coupon is valid
     * @param object $coupon Coupon object
     * @param mixed $discount Discount
     */
    public function coupon_is_valid( $is_valid, $coupon, $discount ) {
        // Skip if admin
        if ( is_admin() ) { return $is_valid; }
        $user_id = get_current_user_id();
        // Check allowed groups
        $groups = $coupon->get_meta( '_wcb2b_coupon_group' );
        if ( ! empty( $groups ) ) {
            // Get current user ID and default group ID
            $user_group_id = get_option( 'wcb2b_guest_group', false );
            // Get group ID
            if ( $group_id = get_the_author_meta( 'wcb2b_group', $user_id ) ) {
                $user_group_id = $group_id;
            }
            if ( ! in_array( $user_group_id, $groups ) ) {
                return false;
            }
        }
        // Check total spent
        $total_spent = $coupon->get_meta( '_wcb2b_coupon_total_spent' );
        if ( ! empty( $total_spent ) ) {
            // Get current user total spent
            $user_total_spent = wc_get_customer_total_spent( $user_id );
            if ( $user_total_spent < $total_spent ) {
                return false;
            }
        }
        return $is_valid;
    }

    /**
     * Process quick order CSV file
     *
     * @return boolean True if process is OK, else False
     */
    public function process_quick_order() {
        if ( isset( $_FILES['wcb2b-quick-order-csv'] ) ) {
            // Check security nonce
            if ( ! isset( $_POST['quick-order-nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['quick-order-nonce'] ), 'quick-order' ) ) {
                wc_add_notice( __( 'An error has occurred, please try again', 'woocommerce-b2b' ), 'error' );
                return false;
            }

            // Check there are no errors
            if ( $_FILES['wcb2b-quick-order-csv']['error'] > 0 ) {
                wc_add_notice( __( 'An error has occurred in upload file, please try again', 'woocommerce-b2b' ), 'error' );
                return false;
            }
            $name = $_FILES['wcb2b-quick-order-csv']['name'];
            $tmp = $_FILES['wcb2b-quick-order-csv']['tmp_name'];
            $ext = pathinfo( $name, PATHINFO_EXTENSION );

            // Check the file is a CSV
            if ( $ext !== 'csv' ) {
                wc_add_notice( __( 'File must be a CSV, please try again', 'woocommerce-b2b' ), 'error' );
                return false;
            }

            if ( ( $fh = fopen( $tmp, 'r' ) ) !== false ) {
                // Necessary if a large CSV file
                set_time_limit( 0 );

                WC()->cart->empty_cart();

                $success = array();
                $errors = array();

                $line = 0;
                while ( ( $data = fgetcsv( $fh, 10000, ';' ) ) !== false ) {
                    $product_sku = sanitize_text_field( $data[0] );
                    if ( empty( $product_sku ) ) {
                        $errors[] = sprintf( __( 'SKU cannot be empty at line %s', 'woocommerce-b2b' ), $line );
                        continue;
                    }
                    $product_quantity = intval( $data[1] );
                    if ( empty( $product_quantity ) || ! is_numeric( $product_quantity ) || $product_quantity <= 0 ) {
                        $errors[] = sprintf( __( 'Product quantity for SKU %s must be a positive integer at line %s', 'woocommerce-b2b' ), $product_sku, $line );
                        continue;
                    }
                    $product_id = wc_get_product_id_by_sku( $product_sku );
                    if ( 0 == $product_id ) {
                        $errors[] = sprintf( __( 'No product with SKU %s was found at line %s', 'woocommerce-b2b' ), $product_sku, $line );
                        continue;
                    }
                    $variation_id = 0;

                    $product = wc_get_product( $product_id );
                    if ( $product->is_type( 'variation' ) ) {
                        $variation_id = $product_id;
                        $product_id = $product->get_parent_id();
                    }

                    $append = '';
                    $min = get_post_meta( $product_id, 'wcb2b_min', true );
                    $max = get_post_meta( $product_id, 'wcb2b_max', true );
                    $step = get_post_meta( $product_id, 'wcb2b_step', true );

                    if ( $step && $product_quantity % $step ) {
                        $append .= sprintf( __( ' - Package quantity %s applied', 'woocommerce-b2b' ), $step );
                        $product_quantity = ceil( round( $product_quantity / 10 ) / $step * 10 ) * $step;
                    }
                    if ( $max && $product_quantity > $max ) {
                        $append .= sprintf( __( ' - Max quantity %s applied', 'woocommerce-b2b' ), $max );
                        $product_quantity = $max;
                    }
                    if ( $min && $product_quantity < $min ) {
                        $append .= sprintf( __( ' - Min quantity %s applied', 'woocommerce-b2b' ), $min );
                        $product_quantity = $min;
                    }

                    WC()->cart->add_to_cart( $product_id, $product_quantity, $variation_id );

                    $success[] = sprintf( __( '%sx %s added to cart', 'woocommerce-b2b' ), $product_quantity, $product_sku ) . $append;
                    $line++;
                }
                fclose( $fh );

                if ( count( $errors ) ) {
                    wc_add_notice( implode( '<br />', $errors ), 'error' );
                }
                if ( count( $success ) ) {
                    wc_add_notice( implode( '<br />', $success ), 'success' );
                }
            }

            wc_maybe_define_constant( 'WCB2B_QUICK_ORDER_PROCESSED', true );

            return true;
        }
    }

    /**
     *  Displays the purchase order number on the order-received page
     *  
     *  @return string
     */
    public function display_purchaseorder_number( $order_id ) {
        if ( $purchaseorder_number = get_post_meta( $order_id, '_wcb2b_purchaseorder_number', true ) ) {
            printf( '<p class="wcb2b-purchaseorder-number"><strong>%s:</strong> %s</p>',
                __( 'Purchase order number', 'woocommerce-b2b' ),
                $purchaseorder_number
            );
        }
    }

    /**
     * Check if customer has reached and exceeded unpaid order amount limit
     */
    public function check_unpaid_order_amount_limit() {
        if ( $customer_id = get_current_user_id() ) {
            $unpaid_amount_limit = get_user_meta( $customer_id, 'wcb2b_unpaid_limit', true );
            $unpaid_amount = wcb2b_get_total_unpaid( $customer_id );
            if ( ! empty( $unpaid_amount_limit ) && $unpaid_amount >= $unpaid_amount_limit ) {
                wc_add_notice( apply_filters( 'wcb2b_unpaid_notice', sprintf( __( 'Sorry, you have reached unpaid total orders amount limit (%s)', 'woocommerce-b2b' ), wc_price( $unpaid_amount_limit ) ), $unpaid_amount_limit ), 'error');
            }
        }
    }

    /**
     * Add invoice email field to billing address (in checkout)
     * 
     * @param array $fields Checkout billing address fields
     * @return array
     */
    public function invoice_email_add_field( $fields ) {
        // Add field exactly after company field
        $fields += array( 'billing_invoice_email' => array(
            'type'          => 'text',
            'label'         => esc_html__( 'Email address for invoices', 'woocommerce-b2b' ),
            'placeholder'   => esc_html_x( 'Email address for invoices', 'placeholder', 'woocommerce-b2b' ),
            'required'      => ( get_option( 'wcb2b_invoice_email_required' ) === 'yes' ),
            'class'         => array( 'form-row-wide' ),
            'clear'         => true,
            'priority'      => 35
        ) );
        return $fields;
    }

    /**
     * Check if invoice email valid in checkout
     * 
     * @param array $fields Checkout billing address fields
     * @param array $errors List of errors
     * @return array
     */
    public function invoice_email_validation_checkout( $fields, $errors ) {
        if ( defined( 'REST_REQUEST' ) ) { return $errors; }

        if ( ! is_email( $fields['billing_invoice_email'] ) ) {
            $errors->add( 'validation', esc_html__( 'Email address for invoices is not a valid email address', 'woocommerce-b2b' ) );
        }
        return $errors;
    }

    /**
     * Add VAT number field to billing address (in checkout)
     * 
     * @param array $fields Checkout billing address fields
     * @return array
     */
    public function vatnumber_add_field( $fields ) {
        // Company is mandatory (only for WC < 3.4.0)
        if ( version_compare( '3.4.0', get_option( 'woocommerce_version' ) ) === 1 ) {
            $fields['billing_company']['required'] = apply_filters( 'wcb2b_billing_company_required', true );
        }

        // Add field exactly after company field
        $fields += array( 'billing_vat' => array(
            'type'          => 'text',
            'label'         => esc_html__( 'Vat number', 'woocommerce-b2b' ),
            'placeholder'   => esc_html_x( 'Vat number', 'placeholder', 'woocommerce-b2b' ),
            'required'      => ( get_option( 'wcb2b_vatnumber_required' ) === 'yes' ),
            'class'         => array( 'form-row-wide' ),
            'clear'         => true,
            'priority'      => 35
        ) );
        return $fields;
    }

    /**
     * Check if VAT number is VIES valid in checkout
     * 
     * @param array $fields Checkout billing address fields
     * @param array $errors List of errors
     * @return array
     */
    public function vatnumber_validation_checkout( $fields, $errors ) {
        if ( defined( 'REST_REQUEST' ) ) { return $errors; }

        if ( ! wcb2b_valid_vies( $fields['billing_country'], $fields['billing_vat'] ) ) {
            $errors->add( 'validation', esc_html__( 'VAT number is not verified by VIES', 'woocommerce-b2b' ) );
        }
        return $errors;
    }

    /**
     * If min amount is not reached, disable go to checkout button (cart)
     */
    public function disable_cart_button() {
        if ( ! is_cart() ) { return; }

        // Check if customer has group, else consider guest
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        $scope = apply_filters( 'wcb2b_min_purchase_amount_scope', 'total' );
        switch ( $scope ) {
            case 'total' :
                $against = WC()->cart->get_total( 'raw' );
                break;
            case 'goods' :
                $against = WC()->cart->get_cart_contents_total();
                break;
        }

        // Check if option is active and minimum amount is reached or not
        $min_price_raw = get_post_meta( $customer_group_id, 'wcb2b_group_min_purchase_amount', true );
        if ( floatval( $min_price_raw ) > floatval( $against ) ) {
            
            // Add a message to inform that minimum amount is not reached yet (in cart)
            if ( get_option( 'wcb2b_display_min_purchase_cart_message' ) === 'yes' ) {
                add_action( 'woocommerce_proceed_to_checkout', function() use ( $min_price_raw ) {
                    $min_price = wc_price( $min_price_raw );
                    echo '<p class="wcb2b_display_min_purchase_cart_message">' . apply_filters( 'wcb2b_display_min_purchase_cart_message', sprintf( esc_html__( 'To proceed to checkout and complete your purchase, make sure you have reached the minimum amount of %s.', 'woocommerce-b2b' ), $min_price ), $min_price_raw, $min_price ) . '<p>';
                }, 20 );

                // Remove "Proceed to checkout" button (also in mini-cart)
                if ( get_option( 'wcb2b_prevent_checkout_button' ) === 'yes' ) {
                    remove_action( 'woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20 );
                    remove_action( 'woocommerce_widget_shopping_cart_buttons', 'woocommerce_widget_shopping_cart_proceed_to_checkout', 20 );
                }
            }
        }
    }

    /**
     * If min amount is not reached, disable go to checkout button (checkout)
     */
    public function disable_checkout_button() {
        if ( ! is_checkout() ) { return; }

        // Check if customer has group, else consider guest
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        $scope = apply_filters( 'wcb2b_min_purchase_amount_scope', 'total' );
        switch ( $scope ) {
            case 'total' :
                $against = WC()->cart->get_total( 'raw' );
                break;
            case 'goods' :
                $against = WC()->cart->get_cart_contents_total();
                break;
        }

        $min_price_raw = get_post_meta( $customer_group_id, 'wcb2b_group_min_purchase_amount', true );
        if ( floatval( $min_price_raw ) > floatval( $against ) ) {
            // Remove terms and conditions
            remove_action( 'woocommerce_checkout_terms_and_conditions', 'wc_checkout_privacy_policy_text', 20 );
            // Block cart payments
            add_filter( 'woocommerce_cart_needs_payment', '__return_false' );
            // Remove payment button
            add_filter( 'woocommerce_order_button_html', '__return_false' );
            // Add a message to inform that minimum amount is not reached yet (in checkout)
            add_action( 'woocommerce_review_order_after_submit', function() use ( $min_price_raw, $against ) {
                $min_price = wc_price( $min_price_raw );
                echo '<p class="wcb2b_display_min_purchase_checkout_message">' . apply_filters( 'wcb2b_display_min_purchase_checkout_message', sprintf( esc_html__( 'To proceed to checkout and complete your purchase, you must reach the minimum amount of %s, but your total cart amount is currently %s. Return to %sshop%s', 'woocommerce-b2b' ), $min_price, wc_price( $against ), '<a href="' . get_permalink( wc_get_page_id( 'shop' ) ) . '">', '</a>' ), $min_price_raw, $min_price ) . '<p>';
            } );
        }
    }

}

return new WCB2B_Frontend_Checkout();