<?php

/**
 * WooCommerce B2B Frontend MyAccount set-up Class
 *
 * @version 3.1.5
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_Frontend_Myaccount Class
 */
class WCB2B_Frontend_Myaccount {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Init current class hooks
     */
    public function init_hooks() {
        if ( get_option( 'wcb2b_enable_quotations' ) === 'yes' ) {
            add_action( 'woocommerce_account_quotations_endpoint', array( $this, 'add_quotations_endpoint_content' ) );
            add_filter( 'woocommerce_endpoint_quotations_title', array( $this, 'add_quotations_endpoint_title' ), 10, 2 );
            add_filter( 'woocommerce_get_endpoint_url', array( $this, 'quotations_pagination' ), 10, 4 );
            add_filter( 'woocommerce_get_query_vars', array( $this, 'add_query_vars' ), 10 );
            add_filter( 'woocommerce_account_menu_items', array( $this, 'add_quotations_menu' ) );
            add_filter( 'woocommerce_account_menu_item_classes', array( $this, 'active_quotations_menu' ), 10, 2 );
            add_filter( 'woocommerce_my_account_my_orders_query', array( $this, 'remove_quotations_from_orders' ), 10, 1 );
            add_action( 'woocommerce_view_order', array( $this, 'add_payment_button' ), 99 );
            add_filter( 'woocommerce_get_order_item_totals', array( $this, 'remove_quotations_payment_method' ), 10, 3 );
            add_filter( 'woocommerce_get_return_url', array( $this, 'redirect_to_quotations' ), 10, 2 );
        }

        if ( get_option( 'wcb2b_show_unpaid_amount' ) === 'yes' ) {
            add_action( 'woocommerce_account_dashboard', array( $this, 'display_unpaid_orders_total' ) );
        }

        if ( get_option( 'wcb2b_show_customer_discount' ) === 'yes' ) {
            add_action( 'woocommerce_account_dashboard', array( $this, 'display_discount' ) );
        }

        if ( get_option( 'wcb2b_extend_registration_form' ) === 'yes' ) {
            add_action( 'woocommerce_register_form_start', array( $this, 'add_extended_fields' ) );
            add_action( 'woocommerce_register_post', array( $this, 'validate_extended_fields' ), 10, 3 );
            add_action( 'woocommerce_created_customer', array( $this, 'save_extended_fields' ) );
            if ( get_option( 'wcb2b_add_invoice_email' ) === 'yes' ) {
                add_action( 'woocommerce_register_post', array( $this, 'invoice_email_validation_registration' ), 11, 3 );
            }
            if ( get_option( 'wcb2b_add_vatnumber' ) === 'yes' ) {
                add_action( 'woocommerce_register_post', array( $this, 'vatnumber_validation_registration' ), 11, 3 );
            }
        }

        if ( get_option( 'wcb2b_registration_notice' ) === 'yes' ) {
            add_action( 'woocommerce_created_customer', array( $this, 'notify_admin' ) );
            add_filter( 'wp_new_user_notification_email_admin', array( $this, 'user_data' ), 10, 3 );
        }

        if ( get_option( 'wcb2b_moderate_customer_registration' ) === 'yes' ) {
            add_action( 'woocommerce_created_customer', array( $this, 'update_user_status' ), 90 );
            add_filter( 'authenticate', array( $this, 'prevent_login' ), 98, 2 );
            add_filter( 'woocommerce_registration_auth_new_customer', array( $this, 'prevent_autologin' ), 98, 2 );
        }

        if ( get_option( 'wcb2b_restricted_catalog' ) === 'yes' ) {
            add_action( 'template_redirect', array( $this, 'restrict_catalog' ), 10 );
            add_filter( 'wp_get_nav_menu_items', array( $this, 'remove_from_menu' ), 10, 3 );
            add_filter( 'get_pages', array( $this, 'remove_from_list' ), 10, 2 );
            add_filter( 'widget_display_callback', array( $this, 'remove_from_widgets' ), 10, 3 );
            add_action( 'init', array( $this, 'remove_from_search' ) );
        }

        if ( get_option( 'wcb2b_show_customer_group' ) === 'yes' ) {
            add_action( 'woocommerce_account_dashboard', array( $this, 'show_customer_group' ) );
        }

        if ( get_option( 'wcb2b_default_group' ) === 'choose' ) {
            add_action( 'woocommerce_register_form_start', array( $this, 'group_add_field' ) );
        }

        add_action( 'woocommerce_created_customer', array( $this, 'assign_customer_group' ) );

        add_action( 'template_redirect', array( $this, 'redirect_login_forms' ), 5 );
    }

    /**
     * Add quotations my-account page endpoint content
     * 
     * @param int $current_page Current page for pagination
     */
    public function add_quotations_endpoint_content( $current_page ) {
        $current_page = empty( $current_page ) ? 1 : absint( $current_page );
        $customer_orders = wc_get_orders( array(
            'customer' => get_current_user_id(),
            'page'     => $current_page,
            'paginate' => true,
            'post_status' => array( 'wc-on-quote', 'wc-quoted' )
        ) );

        wc_get_template(
            'myaccount/orders.php',
            array(
                'current_page'    => absint( $current_page ),
                'customer_orders' => $customer_orders,
                'has_orders'      => 0 < $customer_orders->total,
            )
        );
    }

    /**
     * Add quotations my-account page endpoint content title
     * 
     * @param string $title Page title
     * @param string $endpoint Current endpoint
     * @return string
     */
    public function add_quotations_endpoint_title( $title, $endpoint ) {
        return esc_html__( 'Quotations', 'woocommerce-b2b' );
    }

    /**
     * Adjust quotations pagination URLs
     *   
     * @param string $url Endpoint URL
     * @param string $endpoint Current endpoint name
     * @param integer $value Page number
     * @param string $permalink Current permalink
     * @return string
     */
    public function quotations_pagination( $url, $endpoint, $value, $permalink ) {
        if ( did_action( 'woocommerce_before_account_orders_pagination' ) ) {
            if ( 'orders' == $endpoint && is_wc_endpoint_url( 'quotations' ) ) {
                return wc_get_endpoint_url( 'quotations', $value );
            }
        }
        return $url;
    }

    /**
     * Add query vars to manage endpoint
     * 
     * @param array $query_vars Query vars list
     * @return array
     */
    public function add_query_vars( $query_vars ) {
        $query_vars['quotations'] = get_option( 'wcb2b_account_quotations_endpoint', 'quotations' );
        return $query_vars;
    }

    /**
     * Add quotations endpoint to my-account menu
     * 
     * @param array $menu_links Menu links
     * @return array
     */
    public function add_quotations_menu( $menu_links ) {
        $menu_links = array_slice( $menu_links, 0, 1, true) +
            array( 'quotations' => __( 'Quotations', 'woocommerce-b2b' ) ) +
            array_slice( $menu_links, 1, count( $menu_links ) - 1, true );
        return $menu_links;
    }

    /**
     * Add quotations menu active link management
     * 
     * @param array $classes Current menu link classes
     * @param string $endpoint Current endpoint
     * @return array
     */
    public function active_quotations_menu( $classes, $endpoint ) {
        global $wp;

        if ( isset( $wp->query_vars['view-order'] ) ) {
            $order = wc_get_order( $wp->query_vars['view-order'] );

            if ( 'quotations' === $endpoint && $order->has_status( array( 'quoted', 'on-quote' ) ) ) {
                $classes[] = 'is-active';
            }
            if ( 'orders' === $endpoint && $order->has_status( array( 'quoted', 'on-quote' ) ) ) {
                if ( ( $key = array_search( 'is-active', $classes ) ) !== false ) {
                    unset( $classes[$key] );
                }
            }
        }
        return $classes;
    }

    /**
     * Remove quotations from orders list in my account page
     * 
     * @param array $args Arguments
     * @return array
     */
    public function remove_quotations_from_orders( $args ) {
        $args['post_status'] = array_diff( array_keys( wc_get_order_statuses() ), array( 'wc-on-quote', 'wc-quoted' ) );
        return $args;
    }

    /**
     * Add payment button to quotations
     * 
     * @param integer $order_id Current order ID
     */
    public function add_payment_button( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( $order->has_status( 'quoted' ) ) {
            printf( '<a href="%s" class="button">%s</a>',
                esc_url( $order->get_checkout_payment_url() ),
                __( 'Pay now', 'woocommerce-b2b' )
            );
        }
    }

    /**
     * Prevent quotation display as payment method in order review
     * 
     * @param array $rows Order rows
     * @param object $order Current order instance
     * @param boolean $tax_display Tax display
     * @return array
     */
    public function remove_quotations_payment_method( $rows, $order, $tax_display ) {
        if ( is_object($order) && $order->has_status( array( 'quoted', 'on-quote' ) ) ) {
            unset( $rows['payment_method'] );
        }
        return $rows;
    }

    /**
     * Redirect customers on proper quotation endpoint after placed quotation
     * 
     * @param string $return_url Redirect to URL
     * @param object $order Current order object
     * @return string
     */
    public function redirect_to_quotations( $return_url, $order ) {
        if ( is_object($order) && $order->has_status( 'on-quote' ) ) {
            return wc_get_account_endpoint_url( 'quotations' );
        }
        return $return_url;
    }

    /**
     * Display unpaid orders total in user area
     */
    public function display_unpaid_orders_total() {
        $customer_id = get_current_user_id();
        if ( is_user_logged_in() && wcb2b_has_role( $customer_id, 'customer' ) ) {
            $unpaid_amount_limit = get_user_meta( $customer_id, 'wcb2b_unpaid_limit', true );
            $unpaid_amount = wcb2b_get_total_unpaid( $customer_id );
            if ( ! empty( $unpaid_amount_limit ) ) {
                printf( '<div class="wcb2b-unpaid-total">%s</div><br />',
                    apply_filters( 'wcb2b_unpaid_message' , sprintf( esc_html__( 'Your unpaid amount: %s (Max allowed: %s)', 'woocommerce-b2b' ), 
                        wc_price( $unpaid_amount ),
                        wc_price( $unpaid_amount_limit )
                    ), $customer_id, $unpaid_amount, $unpaid_amount_limit )
                );
            }
        }
    }

    /**
     * Display discount percentage in user area
     */
    public function display_discount() {
        if ( in_array( get_option( 'wcb2b_price_rules' ), array( 'global', 'both' ) ) ) {
            $customer_group_id = get_option( 'wcb2b_guest_group' );
            if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
                if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                    $customer_group_id = $group_id;
                }
            }
            $group = get_post( $customer_group_id );

            // Check if group exists or is deleted precedently
            if ( false === get_post_status( $customer_group_id ) ) { return; }

            // If has discount assigned, display
            if ( $discount = get_post_meta( $group->ID, 'wcb2b_group_discount', true ) ) {
                $discount = number_format( $discount, 
                    wc_get_price_decimals(),
                    wc_get_price_decimal_separator(),
                    wc_get_price_thousand_separator()
                );
                echo '<div class="wcb2b-discount-amount">' . apply_filters( 'wcb2b_discount_message' , sprintf( esc_html__( 'Discount amount assigned to you: %s%%', 'woocommerce-b2b' ), $discount ), $discount ) . '</div><br />';
            }
        }
    }

    /**
     * Add extended fields to customer registration
     */
    public function add_extended_fields() {
        if ( apply_filters( 'wcb2b_extend_registration_form_b2b_only', false ) && is_account_page() ) { return; }

        wp_enqueue_script( 'wc-country-select' );
        wp_enqueue_script( 'wc-address-i18n' );
        wp_enqueue_script( 'wc-checkout' );

        $checkout = WC_Checkout::instance();
        $fields = apply_filters( 'wcb2b_register_form_fields', $checkout->get_checkout_fields( 'billing' ) );

        foreach ( $fields as $key => $field ) {
            // Prevent email field replication
            if ( $key == 'billing_email' ) { continue; }

            if ( isset( $field['country_field'], $fields[ $field['country_field'] ] ) ) {
                $field['country'] = $checkout->get_value( $field['country_field'] );
            }
            woocommerce_form_field( $key, $field, $checkout->get_value( $key ) );
        }
    }

    /**
     * Validate extended fields
     * 
     * @param string $username Post registering username
     * @param string $email Post registering email
     * @param array $errors List of errors
     * @return array
     */
    public function validate_extended_fields( $username, $email, $errors ) {
        if ( defined( 'REST_REQUEST' ) ) { return $errors; }

        $checkout = WC_Checkout::instance();
        $fields = apply_filters( 'wcb2b_register_form_fields', $checkout->get_checkout_fields( 'billing' ) );
        $data = $_POST;

        foreach ( $fields as $key => $field ) {
            if ( $key == 'billing_email' ) { continue; }

            if ( ! isset( $data[ $key ] ) ) {
                continue;
            }

            $required    = ! empty( $field['required'] ) && ! is_user_logged_in(); // WCB2BSA tweak
            $format      = array_filter( isset( $field['validate'] ) ? (array) $field['validate'] : array() );
            $field_label = isset( $field['label'] ) ? $field['label'] : '';

            if ( in_array( 'postcode', $format ) ) {
                $country      = isset( $data[ 'billing_country' ] ) ? $data[ 'billing_country' ] : WC()->customer->{"get_billing_country"}();
                $data[ $key ] = wc_format_postcode( $data[ $key ], $country );

                if ( '' !== $data[ $key ] && ! WC_Validation::is_postcode( $data[ $key ], $country ) ) {
                    switch ( $country ) {
                        case 'IE':
                            /* translators: %1$s: field name, %2$s finder.eircode.ie URL */
                            $postcode_validation_notice = sprintf( __( '%1$s is not valid. You can look up the correct Eircode <a target="_blank" href="%2$s">here</a>.', 'woocommerce-b2b' ), '<strong>' . esc_html( $field_label ) . '</strong>', 'https://finder.eircode.ie' );
                            break;
                        default:
                            /* translators: %s: field name */
                            $postcode_validation_notice = sprintf( __( '%s is not a valid postcode / ZIP.', 'woocommerce-b2b' ), '<strong>' . esc_html( $field_label ) . '</strong>' );
                    }
                    $errors->add( 'validation', apply_filters( 'woocommerce_checkout_postcode_validation_notice', $postcode_validation_notice, $country, $data[ $key ] ), array( 'id' => $key ) );
                }
            }

            if ( in_array( 'phone', $format ) ) {
                $data[ $key ] = wc_format_phone_number( $data[ $key ] );

                if ( '' !== $data[ $key ] && ! WC_Validation::is_phone( $data[ $key ] ) ) {
                    /* translators: %s: phone number */
                    $errors->add( 'validation', sprintf( __( '%s is not a valid phone number.', 'woocommerce-b2b' ), '<strong>' . esc_html( $field_label ) . '</strong>' ), array( 'id' => $key ) );
                }
            }

            if ( in_array( 'email', $format ) && '' !== $data[ $key ] ) {
                $data[ $key ] = sanitize_email( $data[ $key ] );
                if ( ! is_email( $data[ $key ] ) ) {
                    /* translators: %s: email address */
                    $errors->add( 'validation', sprintf( __( '%s is not a valid email address.', 'woocommerce-b2b' ), '<strong>' . esc_html( $field_label ) . '</strong>' ), array( 'id' => $key ) );
                    continue;
                }
            }

            if ( '' !== $data[ $key ] && in_array( 'state', $format ) ) {
                $country      = isset( $data[ 'billing_country' ] ) ? $data[ 'billing_country' ] : WC()->customer->{"get_billing_country"}();
                $valid_states = WC()->countries->get_states( $country );

                if ( ! empty( $valid_states ) && is_array( $valid_states ) && count( $valid_states ) > 0 ) {
                    $valid_state_values = array_map( 'wc_strtoupper', array_flip( array_map( 'wc_strtoupper', $valid_states ) ) );
                    $data[ $key ]       = wc_strtoupper( $data[ $key ] );

                    if ( isset( $valid_state_values[ $data[ $key ] ] ) ) {
                        // With this part we consider state value to be valid as well, convert it to the state key for the valid_states check below.
                        $data[ $key ] = $valid_state_values[ $data[ $key ] ];
                    }

                    if ( ! in_array( $data[ $key ], $valid_state_values ) ) {
                        /* translators: 1: state field 2: valid states */
                        $errors->add( 'validation', sprintf( __( '%1$s is not valid. Please enter one of the following: %2$s', 'woocommerce-b2b' ), '<strong>' . esc_html( $field_label ) . '</strong>', implode( ', ', $valid_states ) ), array( 'id' => $key ) );
                    }
                }
            }

            if ( $required && '' === $data[ $key ] ) {
                /* translators: %s: field name */
                $errors->add( 'required-field', apply_filters( 'woocommerce_checkout_required_field_notice', sprintf( __( '%s is a required field.', 'woocommerce-b2b' ), '<strong>' . esc_html( $field_label ) . '</strong>' ), $field_label ), array( 'id' => $key ) );
            }
        }
        return $errors;
    }

    /**
     * Save extended fields to customer registration
     * 
     * @param int $customer_id Current customer ID
     */
    public function save_extended_fields( $customer_id ) {
        if ( defined( 'REST_REQUEST' ) ) { return true; }

        $checkout = WC_Checkout::instance();
        $fields = apply_filters( 'wcb2b_register_form_fields', $checkout->get_checkout_fields( 'billing' ) );

        foreach ( $fields as $key => $field ) {
            // Prevent email field replication
            if ( $key == 'billing_email' ) { continue; }

            if ( ! isset( $_POST[$key] ) ) {
                continue;
            }

            update_user_meta( $customer_id, $key, sanitize_text_field( $_POST[$key] ) );
            if ( $key == 'billing_first_name' ) {
                update_user_meta( $customer_id, 'first_name', sanitize_text_field( $_POST[$key] ) );
            }
            if ( $key == 'billing_last_name' ) {
                update_user_meta( $customer_id, 'last_name', sanitize_text_field( $_POST[$key] ) );
            }
        }
    }

    /**
     * Send new registration notice to admin by email
     * 
     * @param int $customer_id Current customer ID
     */
    public function notify_admin( $customer_id ) {
        wp_new_user_notification( $customer_id, null, 'admin' );
    }

    /**
     * Add new customer data in admin registration notification
     *
     * @param array $wp_new_user_notification_email_admin Email data
     * @param object $user New user object
     * @param string $blogname WordPress blog name
     */
    public function user_data( $wp_new_user_notification_email_admin, $user, $blogname ) {
        if ( apply_filters( 'wcb2b_extend_new_user_notification_email_admin', false ) ) {
            return $wp_new_user_notification_email_admin;
        }
        $customer = new WC_Customer( $user->ID );
        $address = $customer->get_billing();

        $wp_new_user_notification_email_admin['message'] .= "\r\n\r\n" . WC()->countries->get_formatted_address( $address, "\r\n" ) . "\r\n\r\n";
        if ( $group = get_user_meta( $user->ID, 'wcb2b_group', true ) ) {
            $wp_new_user_notification_email_admin['message'] .= sprintf( __( 'Group: %s', 'woocommerce-b2b' ), $group ) . "\r\n\r\n";
        }
        return $wp_new_user_notification_email_admin;
    }

    /**
     * Update user status on customer creation
     * @param integer $customer_id Current customer ID
     */
    public function update_user_status( $customer_id ) {
        if ( wcb2b_has_role( $customer_id, 'customer' ) ) {
            // By default, consider GUEST if no group
            $customer_group_id = get_option( 'wcb2b_guest_group' );
            if ( $group_id = get_the_author_meta( 'wcb2b_group', $customer_id ) ) {
                $customer_group_id = $group_id;
            }
            // Check group flag
            $moderate_registration = get_post_meta( $customer_group_id, 'wcb2b_group_moderate_registration', true );
            if ( ! $moderate_registration ) {
                update_user_meta( $customer_id, 'wcb2b_status', 1 );
            }
            // Bypass for customers without billing VAT number
            if ( 'yes' == get_option( 'wcb2b_add_vatnumber' ) && apply_filters( 'wcb2b_moderate_customer_registration_only_with_vat', false, $customer_group_id ) ) {
                $billing_vat = get_the_author_meta( 'billing_vat', $customer_id );
                if ( empty( $billing_vat ) ) {
                    // Enable customer status, we suppose is a B2C
                    update_user_meta( $customer_id, 'wcb2b_status', 1 );
                }
            }
        }
    }

    /**
     * Prevent login in wp-login page
     * 
     * @param object $user Current user instance
     * @param string $username Current user username
     * @return object
     */
    public function prevent_login( $user, $username ) {
        if ( ! is_null( $user ) && ! is_wp_error( $user ) ) {
            // If user isn't a customer, skip. Only customers could be inactive
            if ( wcb2b_has_role( $user->ID, 'customer' ) ) {
                $status = get_the_author_meta( 'wcb2b_status', $user->ID );
                if ( 0 == (int)$status ) {
                    $message = __( 'Your account is now under review. You will receive an email as soon as it is activated.', 'woocommerce-b2b' );
                    $user = new WP_Error( 'authentication_failed', $message );
                }
            }
        }
        return $user;
    }

    /**
     * Prevent autologin on new registration
     * 
     * @param boolean $auth Prevent autologin?
     * @param integer $customer_id Current customer ID
     * @return boolean
     */
    public function prevent_autologin( $auth, $customer_id ) {
        if ( ! is_null( $customer_id ) && ! is_wp_error( $customer_id ) ) {
            // If user isn't a customer, skip. Only customers could be inactive
            if ( wcb2b_has_role( $customer_id, 'customer' ) ) {
                $status = get_the_author_meta( 'wcb2b_status', $customer_id );
                if ( 0 == (int)$status ) {
                    // WCB2BSA tweak
                    if ( ! is_user_logged_in() ) {
                        $message = __( 'Your account is now under review. You will receive an email as soon as it is activated.', 'woocommerce-b2b' );
                        wc_add_notice( apply_filters( 'wcb2b_waiting_approvation', $message ), 'notice' );
                    }
                    return false;
                }
            }
        }
        return true;
    }

    /**
     * Check if invoice email is valid in extended fields
     *
     * @param string $username Post registering username
     * @param string $email Post registering email
     * @param array $errors List of errors
     * @return array
     */
    public function invoice_email_validation_registration( $username, $email, $errors ) {
        if ( isset( $_POST['billing_invoice_email'] ) ) {
            if ( ! is_email( $_POST['billing_invoice_email'] ) ) {
                $errors->add( 'validation', esc_html__( 'Email address for invoices is not a valid email address', 'woocommerce-b2b' ) );
            }
        }
        return $errors;
    }

    /**
     * Check if VAT number is VIES valid in extended fields
     *
     * @param string $username Post registering username
     * @param string $email Post registering email
     * @param array $errors List of errors
     * @return array
     */
    public function vatnumber_validation_registration( $username, $email, $errors ) {
        if ( isset( $_POST['billing_country'], $_POST['billing_vat'] ) ) {
            if ( ! wcb2b_valid_vies( $_POST['billing_country'], $_POST['billing_vat'] ) ) {
                $errors->add( 'validation', esc_html__( 'VAT number is not verified by VIES', 'woocommerce-b2b' ) );
            }
        }
        return $errors;
    }

    /**
     * Redirect to login/registration page not logged in customers
     */
    public function restrict_catalog() {
        if ( is_user_logged_in() ) { return; }
        if ( is_woocommerce() || is_cart() || is_checkout() || in_array( get_the_ID(), wcb2b_get_restricted_pages() ) ) {
            $redirect = admin_url();
            if ( $page_id = get_option( 'woocommerce_myaccount_page_id' ) ) {
                $redirect = get_permalink( $page_id );
            }

            wp_safe_redirect( apply_filters( 'wcb2b_restricted_catalog_redirect', $redirect ), 302 );
            exit;
        }
    }

    /**
     * Remove WooCommerce items from menu
     * 
     * @param array $items Menu items
     * @param string $menu Menu name
     * @param array $args Menu arguments
     * @return array
     */
    public function remove_from_menu( $items, $menu, $args ) {
        if ( ! is_user_logged_in() ) {
            $woocommerce_pages = wcb2b_get_restricted_pages();
            foreach ( $items as $key => $item ) {
                if ( in_array( $item->object, array( 'product', 'product_cat', 'product_tag' ) ) || in_array( $item->object_id, $woocommerce_pages ) ) {
                    unset( $items[$key] );
                }
            }
        }
        return $items;
    }

    /**
     * Remove all WooCommerce pages from lists
     * 
     * @param array $pages Pages list
     * @param string $r
     * @return array
     */
    public function remove_from_list( $pages, $r ) {
        if ( did_action( 'init' ) ) {
            if ( ! is_user_logged_in() ) {
                $woocommerce_pages = wcb2b_get_restricted_pages();
                foreach ( $pages as $key => $page ) {
                    if ( in_array( $page->ID, $woocommerce_pages ) ) {
                        unset( $pages[$key] );
                    }
                }
            }
        }
        return $pages;
    }

    /**
     * Remove all WooCommerce widgets
     *
     * @param array $settings Settings list
     * @param string $widget Widget name
     * @param array $args Arguments
     * @return array
     */
    public function remove_from_widgets( $settings, $widget, $args ) {
        if ( ! is_user_logged_in() ) {
            if ( property_exists( $widget, 'widget_id' ) && strpos( $widget->widget_id, 'woocommerce' ) !== false ) {
                return false;
            }
        }
        return $settings;
    }

    /**
     * Remove product post type from search
     */
    public function remove_from_search() {
        if ( ! is_user_logged_in() ) {
            global $wp_post_types;
            if ( post_type_exists( 'product' ) ) {
                $wp_post_types['product']->exclude_from_search = true;
            }
        }
    }

    /**
     * Show customer group in my-account page
     */
    public function show_customer_group() {
        if ( wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $wcb2b_user_group = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $wcb2b_group_name = get_the_title( $wcb2b_user_group );
                echo '<div class="wcb2b-customer-group">' . apply_filters( 'wcb2b_show_group_message', sprintf( __( 'Your current group: %s', 'woocommerce-b2b' ), $wcb2b_group_name ), $wcb2b_group_name ) . '</div>';
            }
        }
    }

    /**
     * Add group field to customer (in registration) when "Allow to chose" enabled
     */
    public function group_add_field() {
        if ( ! is_account_page() ) { return true; }

        $groups = wcb2b_get_groups();
        if ( $groups->found_posts ) {
            woocommerce_form_field( 'wcb2b_group', array(
                'type'          => 'select',
                'class'         => array( 'form-row-wide' ),
                'label'         => esc_html__( 'Choose your group', 'woocommerce-b2b' ),
                'required'      => true,
                'options'       => array_reduce( $groups->posts, function( $result, $item ) {
                    $item = (array)$item;
                    $result[$item['ID']] = $item['post_title'];
                    return $result;
                } )
            ) );
        }
    }

    /**
     * Add new customers to group
     * 
     * @param int $customer_id Current customer ID
     */
    public function assign_customer_group( $customer_id ) { 
        if ( defined( 'REST_REQUEST' ) ) { return true; }

        $wcb2b_group = get_option( 'wcb2b_default_group', false );
        if ( ! empty( $_POST['wcb2b_group'] ) ) {
            $wcb2b_group = (int)$_POST['wcb2b_group'];
        }
        if ( ! empty( $wcb2b_group ) && 'choose' !== $wcb2b_group ) {
            update_user_meta( $customer_id, 'wcb2b_group', $wcb2b_group );

            // To fix caches
            wc_delete_product_transients();
        }
    }

    /**
     * Redirect on default my-account page if customer is logged in and shortcode is custom form
     */
    public function redirect_login_forms() {
        global $post;
        if ( is_user_logged_in() && is_object( $post ) && ( has_shortcode( $post->post_content, 'wcb2b_login_form' ) || has_shortcode( $post->post_content, 'wcb2bloginform' ) ) ) {
            wp_safe_redirect( wc_get_page_permalink( 'myaccount' ) );
             exit;
        }
    }

}

return new WCB2B_Frontend_Myaccount();