<?php

/**
 * WooCommerce B2B Products
 *
 * @version 3.1.5
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_Frontend_Products
 */
class WCB2B_Frontend_Products {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Init current class hooks
     */
    public function init_hooks() {
        add_action( 'wp', array( $this, 'set_query_vars' ), 10 );
        add_filter( 'woocommerce_quantity_input_args', array( $this, 'render_quantity_input' ), 10, 2 );
        add_filter( 'woocommerce_available_variation', array( $this, 'render_quantity_input_variation' ), 10, 3 );
        add_filter( 'woocommerce_loop_add_to_cart_args', array( $this, 'render_quantity_input_loop' ), 10, 2 );
        add_filter( 'woocommerce_after_add_to_cart_form', array( $this, 'display_quantity_notices' ) );

        add_filter( 'woocommerce_variation_prices_regular_price', array( $this, 'calculate_regular_price' ), 10, 2 );
        add_filter( 'woocommerce_product_variation_get_regular_price', array( $this, 'calculate_regular_price' ), 10, 2 );
        add_filter( 'woocommerce_product_get_regular_price', array( $this, 'calculate_regular_price' ), 10, 2 );
        add_filter( 'woocommerce_variation_prices_sale_price', array( $this, 'calculate_sale_price' ), 10, 2 );
        add_filter( 'woocommerce_product_variation_get_sale_price', array( $this, 'calculate_sale_price' ), 10, 2 );
        add_filter( 'woocommerce_product_get_sale_price', array( $this, 'calculate_sale_price' ), 10, 2 );
        add_filter( 'woocommerce_variation_prices_price', array( $this, 'calculate_price' ), 10, 2 );
        add_filter( 'woocommerce_product_variation_get_price', array( $this, 'calculate_price' ), 10, 2 );
        add_filter( 'woocommerce_product_get_price', array( $this, 'calculate_price' ), 10, 2 );
        add_filter( 'woocommerce_get_variation_prices_hash', array( $this, 'prices_hash' ) );

        add_action( 'woocommerce_single_product_summary', array( $this, 'display_tier_prices' ), 15 );
        add_filter( 'woocommerce_available_variation', array( $this, 'display_tier_prices_variation' ) );

        add_filter( 'pre_get_posts', array( $this, 'search_by_sku' ), 15 );

        add_filter( 'woocommerce_product_data_store_cpt_get_products_query', array( $this, 'live_search_prepare' ), 10, 2 );
        add_action( 'wp_ajax_wcb2b_livesearch_results', array( $this, 'live_search_ajax_results' ) );
        add_action( 'wp_ajax_nopriv_wcb2b_livesearch_results', array( $this, 'live_search_ajax_results' ) );
        add_action( 'wp_ajax_wcb2b_livesearch_addtocart', array( $this, 'live_search_ajax_addtocart' ) );
        add_action( 'wp_ajax_nopriv_wcb2b_livesearch_addtocart', array( $this, 'live_search_ajax_addtocart' ) );

        add_filter( 'woocommerce_variation_is_visible', array( $this, 'always_show_product_variation' ), 10, 2);

        add_filter( 'woocommerce_get_stock_html', array( $this, 'hide_stock_by_group' ), 10, 2 );

        add_filter( 'pre_option_woocommerce_price_display_suffix', array( $this, 'price_suffix' ), 99, 3 );

        if ( get_option( 'wcb2b_show_customer_discount_product' ) === 'yes' ) {
            add_action( 'woocommerce_single_product_summary', array( $this, 'display_discount_message' ), 15 );
        }

        if ( get_option( 'wcb2b_hide_prices' ) === 'yes' ) {
            add_action( 'woocommerce_init', array( $this, 'hide_prices_hooks' ) );
        }

        if ( get_option( 'wcb2b_product_cat_visibility' ) === 'yes' ) {
            add_action( 'init', array( $this, 'hide_products_hooks' ) );
            add_filter( 'woocommerce_product_is_visible', array( $this, 'hide_products' ), 10, 2 );
        }

        if ( get_option( 'wcb2b_show_rrp' ) === 'yes' ) {
            add_filter( 'woocommerce_get_price_html', 'wcb2b_display_rrp', 99, 2 );
            add_filter( 'woocommerce_available_variation', 'wcb2b_display_variation_rrp', 99, 3 );
        }

        if ( get_option( 'wcb2b_show_barcode' ) === 'yes' ) {
            add_filter( 'woocommerce_product_meta_start', 'wcb2b_display_barcode', 99, 2 );
            add_filter( 'woocommerce_available_variation', array( $this, 'display_barcode_variation' ), 10, 3 );
        }

        if ( get_option( 'wcb2b_show_already_bought' ) === 'yes' ) {
            add_action( 'woocommerce_after_add_to_cart_button', 'wcb2b_already_bought' );
        }

        if ( get_option( 'wcb2b_show_sales' ) === 'yes' ) {
            add_action( 'woocommerce_product_meta_end', 'wcb2b_product_sales' );
        }

        if ( get_option( 'wcb2b_show_shippings_tab' ) === 'yes' ) {
            add_filter( 'woocommerce_product_tabs', array( $this, 'shippings_tab' ) );
        }
    }

    /**
     * Retrieve product type to fix min and pack settings 
     */
    public function set_query_vars() {
        if ( is_product() ) {
            set_query_var( 'wcb2b_product_type', WC_Product_Factory::get_product_type( get_the_ID() ) );
        }
    }

    /**
     * Add min and pack amounts to WooCommerce frontend
     * 
     * @param array $args Arguments list
     * @param object $product Current product instance
     * @return array
     */
    public function render_quantity_input( $args, $product ) {
        // Product type
        $product_type = get_query_var( 'wcb2b_product_type' );
        // Choose customer group, if not logged in default GUEST
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            // If user is a customer logged in, take his group
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }
        // Packages
        $product_group_packages = get_post_meta( $product->get_ID(), 'wcb2b_product_group_packages', true );
        if ( ! empty( $product_group_packages ) ) {
            if ( is_array( $product_group_packages ) && isset( $product_group_packages[$customer_group_id] ) && ! empty( $product_group_packages[$customer_group_id] ) ) {
                $args['step'] = intval( $product_group_packages[$customer_group_id] );
            }
        }
        // Minimum quantity
        $product_group_min = get_post_meta( $product->get_ID(), 'wcb2b_product_group_min', true );
        if ( ! empty( $product_group_min ) ) {
            if ( is_array( $product_group_min ) && isset( $product_group_min[$customer_group_id] ) && ! empty( $product_group_min[$customer_group_id] ) ) {
                $args['min_value'] = $product_type == 'grouped' ? 0 : intval( $product_group_min[$customer_group_id] );
                if ( ! is_cart() ) {
                    // If not is cart page, force input value to min by default
                    $args['input_value'] = $args['min_value'];
                }
            }
        }
        // Maximum quantity
        $product_group_max = get_post_meta( $product->get_ID(), 'wcb2b_product_group_max', true );
        if ( ! empty( $product_group_max ) ) {
            if ( is_array( $product_group_max ) && isset( $product_group_max[$customer_group_id] ) && ! empty( $product_group_max[$customer_group_id] ) ) {
                if ( -1 == $args['max_value'] || intval( $product_group_max[$customer_group_id] ) < $args['max_value'] ) {
                    $args['max_value'] = intval( $product_group_max[$customer_group_id] );
                }
            }
        }
        return $args;
    }

    /**
     * Add min and pack amounts to WooCommerce frontend [Variables]
     * 
     * @param array $args Arguments list
     * @param object $product Current product instance
     * @param object $variation Current variation instance
     * @return array
     */
    public function render_quantity_input_variation( $args, $product, $variation ) {
        // Choose customer group, if not logged in default GUEST
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            // If user is a customer logged in, take his group
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }
        // Packages
        $product_group_packages = get_post_meta( $variation->get_ID(), 'wcb2b_product_group_packages', true );
        if ( ! empty( $product_group_packages ) ) {
            if ( is_array( $product_group_packages ) && isset( $product_group_packages[$customer_group_id] ) && ! empty( $product_group_packages[$customer_group_id] ) ) {
                $args['step'] = intval( $product_group_packages[$customer_group_id] );
            }
        }
        // Minimum quantity
        $product_group_min = get_post_meta( $variation->get_ID(), 'wcb2b_product_group_min', true );
        if ( ! empty( $product_group_min ) ) {
            if ( is_array( $product_group_min ) && isset( $product_group_min[$customer_group_id] ) && ! empty( $product_group_min[$customer_group_id] ) ) {
                $args['min_qty'] = $args['min_value'] = intval( $product_group_min[$customer_group_id] );
            }
        }
        // Maximum quantity
        $product_group_max = get_post_meta( $variation->get_ID(), 'wcb2b_product_group_max', true );
        if ( ! empty( $product_group_max ) ) {
            if ( is_array( $product_group_max ) && isset( $product_group_max[$customer_group_id] ) && ! empty( $product_group_max[$customer_group_id] ) ) {
                $args['max_qty'] = min( $args['max_qty'], intval( $product_group_max[$customer_group_id] ) );
                if ( -1 == $args['max_qty'] || '' == $args['max_qty'] ) {
                    $args['max_qty'] = intval( $product_group_max[$customer_group_id] );
                }
            }
        }
        $args['quantity_notices'] = wcb2b_get_quantity_notices(
            $args['step'] ?? 1,
            $args['min_qty'] ?? 0,
            $args['max_qty'] ?? 0
        );
        return $args;
    }

    /**
     * Add min and pack amounts to WooCommerce frontend [Variables]
     * 
     * @param array $args Arguments list
     * @param object $product Current product instance
     * @return array
     */
    public function render_quantity_input_loop( $args, $product ) {
        // Choose customer group, if not logged in default GUEST
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            // If user is a customer logged in, take his group
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }
        // Minimum quantity
        $product_group_min = get_post_meta( $product->get_ID(), 'wcb2b_product_group_min', true );
        if ( ! empty( $product_group_min ) ) {
            if ( is_array( $product_group_min ) && isset( $product_group_min[$customer_group_id] ) && ! empty( $product_group_min[$customer_group_id] ) ) {
                $args['quantity'] = intval( $product_group_min[$customer_group_id] );
            }
        }
        return $args;
    }

    /**
     * Add min and pack amount message to product
     */
    public function display_quantity_notices() {
        // Choose customer group, if not logged in default GUEST
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            // If user is a customer logged in, take his group
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        global $product;

        // If is set increment purchase value, display message
        $product_group_packages = get_post_meta( $product->get_id(), 'wcb2b_product_group_packages', true );
        $packages = 1;
        if ( ! empty( $product_group_packages ) ) {
            if ( is_array( $product_group_packages ) && isset( $product_group_packages[$customer_group_id] ) && ! empty( $product_group_packages[$customer_group_id] ) ) {
                $packages = intval( $product_group_packages[$customer_group_id] );
            }
        }

        // If is set max purchase value, display message
        $product_group_max = get_post_meta( $product->get_ID(), 'wcb2b_product_group_max', true );
        $max = 0;
        if ( ! empty( $product_group_max ) ) {
            if ( is_array( $product_group_max ) && isset( $product_group_max[$customer_group_id] ) && ! empty( $product_group_max[$customer_group_id] ) ) {
                $max = intval( $product_group_max[$customer_group_id] );
            }
        }

        // If is set min purchase value, display message
        $product_group_min = get_post_meta( $product->get_id(), 'wcb2b_product_group_min', true );
        $min = 0;
        if ( ! empty( $product_group_min ) ) {
            if ( is_array( $product_group_min ) && isset( $product_group_min[$customer_group_id] ) && ! empty( $product_group_min[$customer_group_id] ) ) {
                $min = intval( $product_group_min[$customer_group_id] );
            }
        }
        printf( '<div id="wcb2b_quantity_notices">%s</div>',
            wcb2b_get_quantity_notices( $packages, $min, $max )
        );
    }

    /**
     * Calculate group regular price
     * 
     * @param string $price Current product price
     * @param object $object Current product/variation instance
     * @return string
     */
    public function calculate_regular_price( $price, $object ) {
        return wcb2b_get_group_price( $price, $object->get_id(), 'regular_price' );
    }

    /**
     * Calculate group sale price
     * 
     * @param string $price Current product price
     * @param object $object Current product/variation instance
     * @return string
     */
    public function calculate_sale_price( $price, $object ) {
        return wcb2b_get_group_price( $price, $object->get_id(), 'sale_price' );
    }

    /**
     * Calculate group final price
     * 
     * @param string $price Current product price
     * @param object $object Current product/variation instance
     * @return string
     */
    public function calculate_price( $price, $object ) {
        return wcb2b_get_group_price( $price, $object->get_id(), 'price' );
    }

    /**
     * Create hash for prices cache
     * 
     * @param string $hash Default hash
     * @return string
     */
    public function prices_hash( $hash ) {
        if ( is_user_logged_in() ) {
            $hash[] = get_current_user_id();
        }
        return $hash;
    }

    /**
     * Display tier prices table
     */
    public function display_tier_prices() {
        if ( get_option( 'wcb2b_hide_prices' ) !== 'yes' || ( get_option( 'wcb2b_hide_prices' ) === 'yes' && is_user_logged_in() ) ) {
            global $product;

            $customer_group_id = get_option( 'wcb2b_guest_group' );
            if ( is_user_logged_in() ) {
                $customer_group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() );
            }

            // Get price visibility
            $product_group_hide_prices = get_post_meta( $product->get_id(), 'wcb2b_product_group_hide_prices', true );
            if ( is_array( $product_group_hide_prices ) && in_array( $customer_group_id, $product_group_hide_prices ) ) {
                return false;
            }

            $product_group_tier_prices = get_post_meta( $product->get_id(), 'wcb2b_product_group_tier_prices', true );
            if ( isset( $product_group_tier_prices[$customer_group_id] ) ) {
                $price = $product->get_price();
                $tier_prices = $product_group_tier_prices[$customer_group_id];

                wc_get_template( 'single-product/tier-price.php', array(
                    'tier_prices' => $tier_prices,
                    'price'       => $price
                ), WCB2B_OVERRIDES, WCB2B_ABSPATH . 'templates/' );
            }
        }
    }

    /**
     * Allow search product by SKU
     *
     * @param object $query WP_Query instance
     */
    public function search_by_sku( $query ) {
        if ( ! is_admin() && $query->is_main_query() && $query->is_search() ) {
            if ( $product_id = wc_get_product_id_by_sku( $query->query['s'] ) ) {
                $product = wc_get_product( $product_id );
                if ( $product->is_type( 'variation' ) ) {
                    $product_id = $product->get_parent_id();
                }
                if ( $product->is_visible() ) {
                    $query->set( 'post__in', array( $product_id ) );
                    unset( $query->query['s'] );
                    unset( $query->query_vars['s'] );
                }
            }
        }
    }

    /**
     * Allow live search products
     *
     * @param object $query WP_Query instance
     * @param array $query_vars WP_Query vars
     * @return object
     */
    public function live_search_prepare( $query, $query_vars ) {
        if ( isset( $query_vars['wcb2b_livesearch'] ) && ! empty( $query_vars['wcb2b_livesearch'] ) ) {
            $query['s'] = esc_attr( $query_vars['wcb2b_livesearch'] );
        }
        return $query;
    }

    /**
     * Ajax call to get Live Search results
     */
    public function live_search_ajax_results() {
        $args = apply_filters( 'wcb2b_livesearch_results_args', array(
            'status'       => array( 'publish' ),
            'type'         => array( 'grouped', 'simple', 'variation' ),
            'limit'        => -1,
            'orderby'      => 'name',
            'order'        => 'ASC',
            'stock_status' => 'instock',
            'return'       => 'ids',
            'exclude'      => wcb2b_get_unallowed_products()
        ) );
        $products_by_sku = wc_get_products( array_merge( $args, array(
            'sku' => esc_attr( $_POST['keyword'] )
        ) ) );

        $products_by_title = wc_get_products( array_merge( $args, array(
            'wcb2b_livesearch' => esc_attr( $_POST['keyword'] )
        ) ) );
        $products = array_unique( array_merge( $products_by_sku, $products_by_title ), SORT_REGULAR );
        if ( empty( $products ) ) { _e( 'No results matching your search', 'woocommerce-b2b' ); }

        foreach ( $products as $product_id ) {
            $product = wc_get_product( $product_id );
            $product_id = $product->get_ID();
            $variation_id = 0;
            $product_meta_id = $product_id;
            if ( $product->get_type() == 'variation' ) {
                $variation_id = $product->get_parent_id();
                $product_meta_id = $variation_id;
            }
            $min = get_post_meta( $product_meta_id, 'wcb2b_min', true );
            wc_get_template( 'global/live-search-form-product.php', array(
                'product'   => $product,
                'data'      => sprintf( 'data-product_id="%s" data-variation_id="%s" data-quantity="%s"',
                    $product_id,
                    $variation_id,
                    $min ? $min : 1
                ),
                'page'      => esc_attr( $_POST['page'] )
            ), WCB2B_OVERRIDES, WCB2B_ABSPATH . 'templates/' );
        }
        die();
    }

    /**
     * Ajax call to add product to cart
     */
    public function live_search_ajax_addtocart() {
        WC_Form_Handler::add_to_cart_action();
        WC_AJAX::get_refreshed_fragments();
        die();
    }

    /**
     * Display tier prices table for variations
     * @param array $variation Choosed variation data
     * @return array
     */
    public function display_tier_prices_variation( $variation ) { 
        if ( get_option( 'wcb2b_hide_prices' ) !== 'yes' || ( get_option( 'wcb2b_hide_prices' ) === 'yes' && is_user_logged_in() ) ) {       
            $customer_group_id = get_option( 'wcb2b_guest_group' );
            if ( is_user_logged_in() ) {
                $customer_group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() );
            }

            // Get price visibility
            $product_group_hide_prices = get_post_meta( $variation['variation_id'], 'wcb2b_product_group_hide_prices', true );
            if ( is_array( $product_group_hide_prices ) && in_array( $customer_group_id, $product_group_hide_prices ) ) {
                return $variation;
            }

            $product_group_tier_prices = get_post_meta( $variation['variation_id'], 'wcb2b_product_group_tier_prices', true );
            if ( isset( $product_group_tier_prices[$customer_group_id] ) ) {
                $price = $variation['display_price'];
                $tier_prices = $product_group_tier_prices[$customer_group_id];
                ob_start();
                wc_get_template( 'single-product/tier-price.php', array(
                    'tier_prices' => $tier_prices,
                    'price'       => $price
                ), WCB2B_OVERRIDES, WCB2B_ABSPATH . 'templates/' );
                $output = ob_get_contents();
                ob_end_clean();
                $variation['price_html'] .= $output;
            }
        }
        return $variation;
    }



    /**
     * Add min and pack amounts to WooCommerce frontend [Variables]
     * 
     * @param array $args Arguments list
     * @param object $product Current product instance
     * @param object $variation Current variation instance
     * @return array
     */
    public function display_barcode_variation( $args, $product, $variation ) {
        if ( $barcode = get_post_meta( $variation->get_id(), 'wcb2b_barcode', true ) ) {
            $args['barcode'] = $barcode;
        }
        return $args;
    }

    /**
     * Always show product variation
     */
    public function always_show_product_variation( $visible, $id ) {
        return apply_filters( 'wcb2b_always_show_product_variation', true, $id );
    }

    /**
     * Hide stock by group
     */
    public function hide_stock_by_group( $html, $product ) {
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() ) {
            $customer_group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() );
        }

        // Get stock visibility
        $product_group_hide_stocks = get_post_meta( $product->get_id(), 'wcb2b_product_group_hide_stocks', true );
        if ( is_array( $product_group_hide_stocks ) && in_array( $customer_group_id, $product_group_hide_stocks ) ) {
            return '';
        }
        return $html;
    }

    /**
     * Change price suffix by group
     *
     * @param mixed $pre_option The value to return instead of the option value
     * @param string $option Option name
     * @param mixed $default The fallback value to return if the option does not exist
     * @return mixed
     */
    public function price_suffix( $pre_option, $option, $default ) {
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() ) {
            $customer_group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() );
        }

        // Get price suffix
        if ( $price_suffix = get_post_meta( $customer_group_id, 'wcb2b_group_price_suffix', true ) ) {
            return $price_suffix;
        }
        return $pre_option;
    }

    /**
     * Display discount percentage in product page
     */
    public function display_discount_message() {
        if ( in_array( get_option( 'wcb2b_price_rules' ), array( 'global', 'both' ) ) ) {
            $customer_group_id = get_option( 'wcb2b_guest_group' );
            if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
                if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                    $customer_group_id = $group_id;
                }
            }
            $group = get_post( $customer_group_id );

            // Check if group exists or is deleted precedently
            if ( false === get_post_status( $customer_group_id ) ) { return; }

            // If has discount assigned, display
            if ( $discount = get_post_meta( $group->ID, 'wcb2b_group_discount', true ) ) {
                $discount = number_format(
                    floatval( $discount ), 
                    wc_get_price_decimals(),
                    wc_get_price_decimal_separator(),
                    wc_get_price_thousand_separator()
                );
                echo '<div class="wcb2b-discount-amount">' . apply_filters( 'wcb2b_discount_message' , sprintf( esc_html__( 'Discount amount assigned to you: %s%%', 'woocommerce-b2b' ), $discount ), $discount ) . '</div><br />';
            }
        }
    }

    /**
     * Hide prices to not logged in users hooks
     */
    public function hide_prices_hooks() {
        if ( ! is_user_logged_in() ) {
            // Remove sale flash and price from loop
            remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_show_product_loop_sale_flash', 10 );
            remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_price', 10 );

            // Remove sale flash and price from single product
            remove_action( 'woocommerce_before_single_product_summary', 'woocommerce_show_product_sale_flash', 10 );
            remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_price', 10 );

            // Remove prices also for grouped and variable products
            add_filter('woocommerce_get_price_html', '__return_false' );

            // Remove add to cart button to variable products
            remove_action( 'woocommerce_single_variation', 'woocommerce_single_variation_add_to_cart_button', 20 );
            
            // Add a link to login page with message
            add_action( 'woocommerce_after_shop_loop_item', 'wcb2b_login_message' );
            add_action( 'woocommerce_single_product_summary', 'wcb2b_login_message' );

            // Any product can be purchased 
            add_filter( 'woocommerce_is_purchasable', '__return_false' );

            // Fix to remove sale flash message from loop
            add_filter( 'woocommerce_sale_flash', '__return_false' );

            // Remove all product from cart
            add_action( 'woocommerce_loaded', function() {
                wc_empty_cart();
            } );

            // Remove offers object if prices are hidden
            add_filter( 'woocommerce_structured_data_product', function( $markup ) {
                unset( $markup['offers'] );
                return $markup;
            } );

            // Remove price filter for guests
            add_action( 'widgets_init', function() {
                unregister_widget( 'WC_Widget_Price_Filter' );
            }, 99 );
        }
    }

    /**
     * Hide unallowed products from search, categories, related, cross-sells, up-sells
     *   
     * @param boolean $visible Product visibility
     * @param integer $product_id Current product ID
     * @return boolean
     */
    public function hide_products( $visible, $product_id ) {
        if ( in_array( $product_id, wcb2b_get_unallowed_products() ) ) {
            return false;
        }
        return $visible;
    }

    /**
     * Remove WooCommerce default hooks to remove hidden products from up-sells, cross-sells, related
     */
    public function hide_products_hooks() {
        if ( current_user_can( 'manage_woocommerce' ) ) { return; }

        WC()->wcb2b_unallowed_terms = wcb2b_set_unallowed_terms();
        WC()->wcb2b_unallowed_products = wcb2b_set_unallowed_products();

        add_filter( 'wp_get_nav_menu_items', array( $this, 'hide_in_menu' ), 10, 3 );
        add_filter( 'woocommerce_products_widget_query_args', array( $this, 'hide_in_widgets' ) );
        add_filter( 'woocommerce_recently_viewed_products_widget_query_args', array( $this, 'hide_in_widgets' ) );
        add_filter( 'get_terms_args', array( $this, 'hide_product_categories' ), 10, 2 );
        add_action( 'template_redirect', array( $this, 'hidden_product_redirect' ) );
    }

    /**
     * Exclude not allowed product categories and products
     *
     * @param array $items Menu items
     * @param string $menu Menu name
     * @param array $args Menu arguments
     * @return array
     */
    public function hide_in_menu( $items, $menu, $args ) {
        $unallowed_products = wcb2b_get_unallowed_products();
        $unallowed_terms = wcb2b_get_unallowed_terms();

        if ( is_array( $items ) && ! empty( $items ) ) {
            foreach ( $items as $key => $item ) {
                if (  
                        ( $item->object == 'product' && in_array( $item->object_id, $unallowed_products ) ) || 
                        ( $item->object == 'product_cat' && in_array( $item->object_id, $unallowed_terms ) )
                    ) {
                    unset( $items[$key] );
                }
            }
        }
        return $items;
    }

    /**
     * Remove products from WooCommerce products widgets and shortcodes if has restricted category
     * 
     * @param  array $args Arguments
     * @return array
     */
    public function hide_in_widgets( $args ) {
        $args['tax_query'][] = array(
            'taxonomy'          => 'product_cat',
            'field'             => 'term_id',
            'terms'             => wcb2b_get_unallowed_terms(),
            'operator'          => 'NOT IN',
            'include_children'  => false
        );
        return $args;
    }

    /**
     * Exclude globally not allowed product categories from displaying
     *   
     * @param array $args Arguments
     * @param array $taxonomies Taxonomies
     * @return array
     */
    public function hide_product_categories( $args, $taxonomies ) {
        if ( ! isset( $args['taxonomy'] ) ) { return $args; }
        if ( ! in_array( 'product_cat', $args['taxonomy'] ) || count( $args['taxonomy'] ) > 1 ) { return $args; }

        $unallowed_terms = wcb2b_get_unallowed_terms();
        if ( ! empty( $args['exclude'] ) ) {
            $args['exclude'] = array_merge( $args['exclude'], $unallowed_terms );
        } else {
            $args['exclude'] = $unallowed_terms;
        }
        return $args;
    }

    /**
     * Redirect to choosed page if product has restricted category
     */
    public function hidden_product_redirect() {
        global $wp_query;
        $unallowed_terms = wcb2b_get_unallowed_terms();

        // If is a product page
        if ( is_product() ) {
            $skip = empty( $unallowed_terms ) || ! has_term( $unallowed_terms, 'product_cat', $wp_query->post->ID );
        } elseif ( isset( $wp_query->query['product_cat'] ) ) {
            $terms = get_terms( array(
                'taxonomy'  => 'product_cat',
                'slug'      => $wp_query->query['product_cat'],
                'number'    => 1,
                //'include' => 'all'
            ) );

            if ( is_wp_error( $terms ) || empty( $terms ) ) {
                $skip = true;
            } else {
                $term = array_shift( $terms );
                $skip = ! in_array( $term->term_id, $unallowed_terms );
            }
        } else {
            $skip = true;
        }

        if ( $skip ) { return; }

        $redirect = get_option( 'wcb2b_redirect_not_allowed', 'null' );
        switch ( $redirect ) {
            case 'null' :
                break;
            case '0' :
                $wp_query->set_404();
                status_header( 404 );
                break;
            default :
                wp_safe_redirect( get_permalink( $redirect ), 302 );
                exit;
                break;
        }
    }

    /**
     * Add shipping tab in product page
     *
     * @param array $tabs All product tabs
     * @return array
     */
    public function shippings_tab( $tabs ) {
        $tabs['desc_tab'] = array(
            'title'     => __( 'Shipping Informations', 'woocommerce-b2b' ),
            'priority'  => 50,
            'callback'  => 'wcb2b_shipping_table_content'
        );
        return $tabs;
    }

}

return new WCB2B_Frontend_Products();