<?php

/**
 * WooCommerce B2B Common hooks set-up
 *
 * @version 3.1.4
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_Hooks Class
 */
class WCB2B_Hooks {

    /**
     * Constructor
     */
    public function __construct() {
        $this->includes();
        $this->init_hooks();
    }

    /**
     * Init current class hooks
     */
    public function init_hooks() {
        add_filter( 'is_protected_meta', array( $this, 'hide_protected_meta' ), 10, 3 );

        add_action( 'init', array( $this, 'tax_exemption' ) );
        add_action( 'init', array( $this, 'tax_exemption_by_country' ), 11 );
        add_action( 'woocommerce_new_order', array( $this, 'add_customer_group' ), 10, 2 );

        add_action( 'init', array( $this, 'add_quotations_endpoint' ) );
        add_filter( 'woocommerce_payment_gateways', array( $this, 'add_gateways' ) );
        add_action( 'woocommerce_register_shop_order_post_statuses', array( $this, 'register_order_statuses' ) );
        add_filter( 'wc_order_statuses', array( $this, 'add_order_statuses' ) );
        add_filter( 'wc_order_is_editable', array( $this, 'make_order_statuses_editable' ), 10, 2 );
        add_filter( 'woocommerce_email_actions', array( $this, 'set_emails_actions' ) );
        add_filter( 'woocommerce_email_classes', array( $this, 'set_quotations_emails' ) );
        add_filter( 'woocommerce_email_recipient_new_order', array( $this, 'prevent_new_order_email' ), 20, 2 );
        add_filter( 'woocommerce_email_order_meta_fields', array( $this, 'display_purchaseorder_number' ), 10, 3 );
        add_filter( 'woocommerce_can_reduce_order_stock', array( $this, 'prevent_stock_reduction_on_quotation' ), 10, 2 );
        add_action( 'woocommerce_order_status_changed', array( $this, 'stock_reduction_on_quotation' ), 20, 4 );
        add_filter( 'woocommerce_valid_order_statuses_for_payment', array( $this, 'quotations_can_be_payed' ) );
        add_filter( 'woocommerce_valid_order_statuses_for_cancel', array( $this, 'quotations_can_be_cancelled' ) );

        if ( get_option( 'wcb2b_moderate_customer_registration' ) === 'yes' ) {
            add_filter( 'woocommerce_email_classes', array( $this, 'pending_account_email' ) );
            add_action( 'woocommerce_email_footer', array( $this, 'new_account_email_message' ) );
        }

        if ( get_option( 'wcb2b_tax_display' ) === 'yes' ) {
            add_filter( 'pre_option_woocommerce_tax_display_shop', array( $this, 'tax_split' ), 99, 3 );
            add_filter( 'pre_option_woocommerce_tax_display_cart', array( $this, 'tax_split' ), 99, 3 );
        }

        if ( get_option( 'wcb2b_add_invoice_email' ) === 'yes' ) {
            add_filter( 'woocommerce_formatted_address_replacements', array( $this, 'invoice_email_add_to_email' ), 10, 2 );
            add_filter( 'woocommerce_localisation_address_formats', array( $this, 'invoice_email_format' ) );
            add_filter( 'woocommerce_my_account_my_address_formatted_address', array( $this, 'invoice_email_add_to_address' ), 10, 3 );
            add_filter( 'woocommerce_order_formatted_billing_address', array( $this, 'invoice_email_add_to_order_details' ), 10, 2 );
        }

        if ( get_option( 'wcb2b_add_vatnumber' ) === 'yes' ) {
            add_filter( 'woocommerce_formatted_address_replacements', array( $this, 'vatnumber_add_to_email' ), 10, 2 );
            add_filter( 'woocommerce_localisation_address_formats', array( $this, 'vatnumber_format' ) );
            add_filter( 'woocommerce_my_account_my_address_formatted_address', array( $this, 'vatnumber_add_to_address' ), 10, 3 );
            add_filter( 'woocommerce_order_formatted_billing_address', array( $this, 'vatnumber_add_to_order_details' ), 10, 2 );
        }
    }

    /**
     * Include any classes we need
     */
    public function includes() {
        add_action( 'plugins_loaded', function() {
            include_once WCB2B_ABSPATH . 'includes/classes/gateways/class-wcb2b-payment-gateway-invoice.php';
            include_once WCB2B_ABSPATH . 'includes/classes/gateways/class-wcb2b-payment-gateway-purchaseorder.php';
            include_once WCB2B_ABSPATH . 'includes/classes/gateways/class-wcb2b-payment-gateway-quotation.php';
        }, 11 );
    }

    /**
     * Hide protected meta$protected, $meta_key, $meta_type
     *
     * @param boolean $protected Is meta protected?
     * @param string $meta_key Meta key name
     * @param string $meta_type Meta type
     * @return boolean
     */
    public function hide_protected_meta( $protected, $meta_key, $meta_type ) {
        $meta = array(
            'wcb2b_product_group_packages',
            'wcb2b_product_group_min',
            'wcb2b_product_group_max',
            'wcb2b_barcode',
            'wcb2b_group_discount',
            'wcb2b_product_group_prices',
            'wcb2b_product_group_tier_prices',
            'wcb2b_group_min_purchase_amount',
            'wcb2b_group_tax_exemption',
            'wcb2b_group_packaging_fee',
            'wcb2b_group_gateways',
            'wcb2b_group_shippings',
            'wcb2b_group_terms_conditions',
            'wcb2b_group_visibility',
            'wcb2b_coupon'
        );
        if ( in_array( $meta_key, $meta ) ) {
            $protected = true;
        }
        return $protected;
    }

    /**
     * Add customer group to order
     *
     * @param int $order_id Current order ID
     * @param object $order Current order instance
     */
    public function add_customer_group( $order_id, $order ) {
        $group_id = get_option( 'wcb2b_guest_group', 0 );
        if ( $customer_group_id = get_the_author_meta( 'wcb2b_group', $order->get_customer_id() ) ) {
            $group_id = $customer_group_id;
        }
        update_post_meta( $order->get_id(), '_wcb2b_group', $group_id );
    }

    /**
     * Add quotations my-account page endpoint
     */
    public function add_quotations_endpoint() {
        add_rewrite_endpoint( get_option( 'wcb2b_account_quotations_endpoint', 'quotations' ), EP_ROOT | EP_PAGES );
        flush_rewrite_rules();
    }

    /**
     * Add gateways (quotation, invoice)
     * 
     * @param array $gateways All payment gateways
     * @return array
     */
    public function add_gateways( $gateways ) {
        $gateways[] = 'WCB2B_Gateway_Invoice';
        $gateways[] = 'WCB2B_Gateway_PurchaseOrder';
        if ( ! is_wc_endpoint_url( 'order-pay' ) ) {
            $gateways[] = 'WCB2B_Gateway_Quotation';
        }
        return $gateways;
    }

    /**
     * Register new order statuses
     * 
     * @param array $order_statuses All statuses settings list
     * @return array
     */
    public function register_order_statuses( $order_statuses ) {
        $order_statuses['wc-invoice-payment'] = array(
            'label'                     => __( 'Awaiting invoice payment', 'woocommerce-b2b' ),
            'public'                    => true,
            'show_in_admin_status_list' => true,
            'show_in_admin_all_list'    => true,
            'exclude_from_search'       => false,
            'label_count'               => _n_noop( 'Awaiting invoice payment <span class="count">(%s)</span>', 'Awaiting invoice payment <span class="count">(%s)</span>', 'woocommerce-b2b' )
        );
        $order_statuses['wc-on-quote'] = array(
            'label'                     => __( 'On quote', 'woocommerce-b2b' ),
            'public'                    => true,
            'show_in_admin_status_list' => true,
            'show_in_admin_all_list'    => true,
            'exclude_from_search'       => false,
            'label_count'               => _n_noop( 'On quote <span class="count">(%s)</span>', 'On quote <span class="count">(%s)</span>', 'woocommerce-b2b' )
        );
        $order_statuses['wc-quoted'] = array(
            'label'                     => __( 'Quoted', 'woocommerce-b2b' ),
            'public'                    => true,
            'show_in_admin_status_list' => true,
            'show_in_admin_all_list'    => true,
            'exclude_from_search'       => false,
            'label_count'               => _n_noop( 'Quoted <span class="count">(%s)</span>', 'Quoted <span class="count">(%s)</span>', 'woocommerce-b2b' )
        );
        return $order_statuses;
    }

    /**
     * Add new statuses to list
     * 
     * @param array $order_statuses All statuses list
     * @return array
     */
    public function add_order_statuses( $order_statuses ) {
        $order_statuses['wc-invoice-payment'] = __( 'Awaiting invoice payment', 'woocommerce-b2b' );
        $order_statuses['wc-on-quote'] = __( 'On quote', 'woocommerce-b2b' );
        $order_statuses['wc-quoted'] = __( 'Quoted', 'woocommerce-b2b' );
        return $order_statuses;
    }

    /**
     * Make order statuses editable
     * 
     * @param boolean $editable Order can be edit?
     * @param object $order Current order instance
     * @return boolean
     */
    public function make_order_statuses_editable( $editable, $order ) {
        if ( in_array( $order->get_status(), array( 'on-quote' ) ) ) {
            return true;
        }
        return $editable;
    }

    /**
     * Add transactional email for quotations
     * 
     * @param array $emails Emails settings
     * @return array
     */
    public function set_quotations_emails( $emails ) {
        $emails['WCB2B_Email_New_Quote']   = include WCB2B_ABSPATH . 'includes/classes/emails/class-wcb2b-email-new-quote.php';
        $emails['WCB2B_Email_Customer_OnQuote_Order']   = include WCB2B_ABSPATH . 'includes/classes/emails/class-wcb2b-email-customer-onquote-order.php';
        $emails['WCB2B_Email_Customer_Quoted_Order']    = include WCB2B_ABSPATH . 'includes/classes/emails/class-wcb2b-email-customer-quoted-order.php';
        return $emails;
    }

    /**
     * Prevent sending new order email if is a quotation
     * 
     * @param string $recipient Email recipient
     * @param object $order Current order object
     * @return string
     */
    public function prevent_new_order_email( $recipient, $order ) {
        if ( is_object( $order ) && $order->has_status( 'on-quote' ) ) {
            return '';
        }
        return $recipient;
    }

    /**
     *  Adds the purchase order number to the order emails
     *  
     *  @param array $fields The order meta fields
     *  @param bool $sent_to_admin Send email to admin as well as customer?
     *  @param object $order The order object
     *  
     *  @return array
     */
    public function display_purchaseorder_number( $fields, $sent_to_admin = true, $order ) {
        if ( $purchaseorder_number = get_post_meta( $order->get_id(), '_wcb2b_purchaseorder_number', true ) ) {
            $fields['wcb2b_purchaseorder_number'] = array(
                'label' => __( 'Purchase Order number', 'woocommerce-b2b' ), 
                'value' => esc_html( $purchaseorder_number )
            );
        }
        return $fields;
    }

    /**
     * Add transactional email actions for quotations
     * 
     * @param array $emails Emails settings
     * @return array
     */
    public function set_emails_actions( $emails ) {
        $emails[] = 'woocommerce_order_status_pending_to_on-quote';
        $emails[] = 'woocommerce_order_status_on-quote';
        $emails[] = 'woocommerce_order_status_quoted';
        return $emails;
    }

    /**
     * Prevent stock reduction for quotations
     * 
     * @param boolean $reduce_stock If reduce or not stock
     * @param object $order Current order instance
     * @return boolean
     */
    public function prevent_stock_reduction_on_quotation( $reduce_stock, $order ) {
        if ( $order->has_status( 'on-quote' ) ) {
            $reduce_stock = false;
        }
        return $reduce_stock;
    }

    /**
     * Proceed to reduce stock on quotation completed
     * 
     * @param integer $order_id Current order ID
     * @param string $old_status Order status before change
     * @param string $new_status Order status after change
     * @param object $order Current order instance
     */
    public function stock_reduction_on_quotation( $order_id, $old_status, $new_status, $order ){
        // Only for 'processing' and 'completed' order statuses change
        if ( $new_status == 'quoted' ) {
            $stock_reduced = get_post_meta( $order_id, '_order_stock_reduced', true );
            if ( empty( $stock_reduced ) ) {
                wc_reduce_stock_levels( $order_id );
            }
        }
        if ( $new_status == 'invoice-payment' ) {
            // Getting all WC_emails objects
            $emails = WC()->mailer()->get_emails();
            $emails['WC_Email_Customer_On_Hold_Order']->trigger( $order_id ); // Send to customer
            $emails['WC_Email_New_Order']->trigger( $order_id ); // Send to admin
        }
    }

    /**
     * Mark quotations as payable
     * 
     * @param array $statuses Payable statuses
     * @return array
     */
    public function quotations_can_be_payed( $statuses ) {
        $statuses[] = 'quoted';
        return $statuses;
    }

    /**
     * Mark quotations as cancellable
     * 
     * @param array $statuses Cancellable statuses
     * @return array
     */
    public function quotations_can_be_cancelled( $statuses ) {
        $statuses[] = 'on-quote';
        $statuses[] = 'quoted';
        return $statuses;
    }

    /**
     * Include email classes
     * 
     * @param array $emails Email classes list
     * @return array
     */
    public function pending_account_email( $emails ) {
        if ( ! array_key_exists( 'WCB2B_Email_Customer_Status_Notification', $emails ) ) {
            $emails['WCB2B_Email_Customer_Status_Notification'] = include_once WCB2B_ABSPATH . 'includes/classes/emails/class-wcb2b-email-customer-status-notification.php';
        }

        return $emails;
    }

    /**
     * Add message into new account email to inform customers to wait activation
     * 
     * @param object $email Current email instance
     */
    public function new_account_email_message( $email ) {
        // Customize new account email
        if ( is_object( $email ) && $email->id == 'customer_new_account' ) {
            $status = get_the_author_meta( 'wcb2b_status', $email->object->ID );
            if ( 0 == (int)$status ) {
                echo '<p>' . apply_filters( 'wcb2b_new_account_email', __( 'We are checking your account, please wait for the activation confirmation email before trying to login', 'woocommerce-b2b' ) ) . '</p>';
            }
        }
    }

    /**
     * Apply tax exemption for groups
     *
     * @param string $tax_class Tax class to apply
     * @param object $product Current product instance
     * @return string
     */
    public function tax_exemption() {
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        if ( isset( WC()->customer ) ) {
            if ( get_post_meta( $customer_group_id, 'wcb2b_group_tax_exemption', true ) ) {
                WC()->customer->set_is_vat_exempt( true );
            } else {
                WC()->customer->set_is_vat_exempt( false );
            }
        }
    }

    /**
     * Apply tax free by country
     * 
     * @param  array $data Posted data
     */
    public function tax_exemption_by_country() {
        // Retrieve guest group
        $guest_group_id = $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            // Check if customer is logged and has a group
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
        }

        if ( isset( WC()->customer ) ) {
            if ( get_post_meta( $customer_group_id, 'wcb2b_group_tax_exemption', true ) ) {
                WC()->customer->set_is_vat_exempt( true );
            } else {
                WC()->customer->set_is_vat_exempt( false );
                // Skip if customer group is guest
                if ( $guest_group_id !== $customer_group_id ) {
                    // Get tax free countries
                    $countries = get_option( 'wcb2b_tax_exemption_countries' );
                    // Check if country is tax free
                    if ( is_array( $countries ) && in_array( WC()->customer->get_billing_country(), $countries ) ) {
                        WC()->customer->set_is_vat_exempt( true );
                    }
                }
            }
        }
    }

    /**
     * Force to show prices with taxes to guest customers and without taxes to customers inserted into a group
     * 
     * @param mixed $pre_option The value to return instead of the option value
     * @param string $option Option name
     * @param mixed $default The fallback value to return if the option does not exist
     * @return mixed
     */
    public function tax_split( $pre_option, $option, $default ) {
        // By default, consider guest group configuration
        $customer_group_id = get_option( 'wcb2b_guest_group' );
        if ( is_user_logged_in() && wcb2b_has_role( get_current_user_id(), 'customer' ) ) {
            // If user is a customer logged in, take his group
            if ( $group_id = get_the_author_meta( 'wcb2b_group', get_current_user_id() ) ) {
                $customer_group_id = $group_id;
            }
            // Get and return group option for tax display (if exists, else default)
            if ( $option = get_post_meta( $customer_group_id, 'wcb2b_group_tax_display', true ) ) {
                return $option;
            }
        }
        return $pre_option;
    }

    /**
     * Creating merger invoice email variables for printing formatting
     * 
     * @param array $address Address fields
     * @param array $args Arguments
     * @return array
     */
    public function invoice_email_add_to_email( $address, $args ) {
        $address['{invoice_email}'] = '';
        $address['{invoice_email_upper}'] = '';

        if ( ! empty( $args['invoice_email'] ) ) {
            $address['{invoice_email}'] = $args['invoice_email'];
            $address['{invoice_email_upper}'] = strtoupper($args['invoice_email']);
        }
        return $address;
    }

    /**
     * Redefine the formatting to print the address, including invoice email
     * 
     * @param string $formats Current i18n format
     * @return string
     */
    public function invoice_email_format( $formats ) {
        return str_replace( "{company}", "{company}\n{invoice_email}", $formats );
    }

    /**
     * Add invoice email field to billing address (in My account)
     * 
     * @param array $fields Checkout billing address fields
     * @param int $customer_id Current customer ID
     * @param string $type Address type (billing/shipping)
     * @return array
     */
    public function invoice_email_add_to_address( $fields, $customer_id, $type ) {
        if ( $type == 'billing' ) {
            $fields['invoice_email'] = get_user_meta( $customer_id, 'billing_invoice_email', true );
        }
        return $fields;
    }

    /**
     * Add invoice email field to billing address (in Order received)
     * 
     * @param array $fields Checkout billing address fields
     * @param object $order Current order object instance
     * @return array
     */
    public function invoice_email_add_to_order_details( $fields, $order ) {
        $fields['invoice_email'] = get_post_meta( $order->get_id(), '_billing_invoice_email', true );
        return $fields;
    }

    /**
     * Creating merger VAT number variables for printing formatting
     * 
     * @param array $address Address fields
     * @param array $args Arguments
     * @return array
     */
    public function vatnumber_add_to_email( $address, $args ) {
        $address['{vat}'] = '';
        $address['{vat_upper}'] = '';

        if ( ! empty( $args['vat'] ) ) {
            $address['{vat}'] = $args['vat'];
            $address['{vat_upper}'] = strtoupper($args['vat']);
        }
        return $address;
    }

    /**
     * Redefine the formatting to print the address, including VAT number
     * 
     * @param string $formats Current i18n format
     * @return string
     */
    public function vatnumber_format( $formats ) {
        return str_replace( "{company}", "{company}\n{vat_upper}", $formats );
    }

    /**
     * Add VAT number field to billing address (in My account)
     * 
     * @param array $fields Checkout billing address fields
     * @param int $customer_id Current customer ID
     * @param string $type Address type (billing/shipping)
     * @return array
     */
    public function vatnumber_add_to_address( $fields, $customer_id, $type ) {
        if ( $type == 'billing' ) {
            $fields['vat'] = get_user_meta( $customer_id, 'billing_vat', true );
        }
        return $fields;
    }

    /**
     * Add VAT number field to billing address (in Order received)
     * 
     * @param array $fields Checkout billing address fields
     * @param object $order Current order object instance
     * @return array
     */
    public function vatnumber_add_to_order_details( $fields, $order ) {
        $fields['vat'] = get_post_meta( $order->get_id(), '_billing_vat', true );
        return $fields;
    }

}

return new WCB2B_Hooks();