<?php

/**
 * WooCommerce B2B Post types set-up Class
 *
 * @version 3.1.5
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_Post_Types Class
 */
class WCB2B_Post_Types {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Init current class hooks
     */
    public function init_hooks() {
        add_action( 'init', array( $this, 'register_group_post_type' ) );
        add_filter( 'post_row_actions', array( $this, 'row_actions' ), 100, 2 );
        add_action( 'add_meta_boxes_wcb2b_group', array( $this, 'add_metabox' ) );
        add_action( 'save_post_wcb2b_group', array( $this, 'save_meta_box_fields' ) );
        add_filter( 'manage_wcb2b_group_posts_columns', array( $this, 'render_columns' ) );
        add_filter( 'manage_wcb2b_group_posts_custom_column', array( $this, 'render_columns_value' ), 10, 2 );
        add_filter( 'manage_edit-wcb2b_group_sortable_columns', array( $this, 'make_columns_sortable' ) );
        add_filter( 'default_hidden_columns', array( $this, 'default_hidden_columns' ), 10, 2 );
        add_action( 'bulk_edit_custom_box',  array( $this, 'quick_edit_fields' ), 10, 2 );
        add_filter( 'post_row_actions', array( $this, 'remove_guest_group_delete' ), 10, 2 );
        add_filter( 'user_has_cap', array( $this, 'prevent_guest_group_delete' ), 10, 3 );
        add_action( 'updated_option', array( $this, 'updated_group' ), 10, 3 );
        add_action( 'transition_post_status', array( $this, 'updated_group_status' ), 10, 3 );
        add_action( 'trash_wcb2b_group', array( $this, 'deleted_group' ) );
    }

    /**
     * Register custom group post type
     */
    public function register_group_post_type() {
        $labels = array(
            'name'                  => __( 'Groups', 'woocommerce-b2b' ),
            'singular_name'         => __( 'Group', 'woocommerce-b2b' ),
            'all_items'             => __( 'All Groups', 'woocommerce-b2b' ),
            'menu_name'             => _x( 'Groups', 'Admin menu name', 'woocommerce-b2b' ),
            'add_new'               => __( 'Add New', 'woocommerce-b2b' ),
            'add_new_item'          => __( 'Add new group', 'woocommerce-b2b' ),
            'edit'                  => __( 'Edit', 'woocommerce-b2b' ),
            'edit_item'             => __( 'Edit group', 'woocommerce-b2b' ),
            'new_item'              => __( 'New group', 'woocommerce-b2b' ),
            'view_item'             => __( 'View group', 'woocommerce-b2b' ),
            'view_items'            => __( 'View groups', 'woocommerce-b2b' ),
            'search_items'          => __( 'Search groups', 'woocommerce-b2b' ),
            'not_found'             => __( 'No groups found', 'woocommerce-b2b' ),
            'not_found_in_trash'    => __( 'No groups found in trash', 'woocommerce-b2b' ),
            'parent'                => __( 'Parent group', 'woocommerce-b2b' ),
            'featured_image'        => __( 'Group image', 'woocommerce-b2b' ),
            'set_featured_image'    => __( 'Set group image', 'woocommerce-b2b' ),
            'remove_featured_image' => __( 'Remove group image', 'woocommerce-b2b' ),
            'use_featured_image'    => __( 'Use as group image', 'woocommerce-b2b' ),
            'insert_into_item'      => __( 'Insert into group', 'woocommerce-b2b' ),
            'uploaded_to_this_item' => __( 'Uploaded to this group', 'woocommerce-b2b' ),
            'filter_items_list'     => __( 'Filter groups', 'woocommerce-b2b' ),
            'items_list_navigation' => __( 'Groups navigation', 'woocommerce-b2b' ),
            'items_list'            => __( 'Groups list', 'woocommerce-b2b' )
        );
        $args = array(
            'label'                 => __( 'Group', 'woocommerce-b2b' ),
            'description'           => __( 'This is where you can add new groups to your customers.', 'woocommerce-b2b' ),
            'labels'                => $labels,
            'supports'              => array( 'title' ),
            'hierarchical'          => false,
            'public'                => false,
            'show_ui'               => true,
            'show_in_menu'          => 'users.php',
            'menu_position'         => 99,
            'show_in_admin_bar'     => true,
            'show_in_nav_menus'     => false,
            'can_export'            => true,
            'has_archive'           => false,
            'exclude_from_search'   => true,
            'publicly_queryable'    => false,
            'capability_type'       => 'product',
            'map_meta_cap'          => true
        );
        register_post_type( 'wcb2b_group', $args );
    }

    /**
     * Add Group ID to list table rows
     */
    public function row_actions( $actions, $post ) {
        if ( 'wcb2b_group' === $post->post_type ) {
            return array_merge( array( 'id' => sprintf( __( 'ID: %d', 'woocommerce-b2b' ), $post->ID ) ), $actions );
        }
        return $actions;
    }

    /**
     * Add group post type metabox with custom fields
     * 
     * @param WP_Post $post The current WP_Post object
     */
    public function add_metabox( $post ) {
        // Discount meta box
        add_meta_box( 'wcb2b_group-discount-meta_box', __( 'Discount (%)', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-discount-nonce' );

            // Retrieve group discount current value
            $group_discount = get_post_meta( $post->ID, 'wcb2b_group_discount', true );

            // Display fields
            printf( '<p><input type="text" name="wcb2b_group_discount" value="%s" pattern="[0-9]+([\.][0-9]+)?" title="' . __( 'This should be a number with up to 2 decimal places, with (.) as decimal separator', 'woocommerce-b2b' ) . '" /></p>',
                $group_discount
            );
        }, 'wcb2b_group', 'side', 'low' );

        // Discount meta box
        add_meta_box( 'wcb2b_group-packaging_fee-meta_box', __( 'Packaging fee', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-packaging_fee-nonce' );

            // Retrieve group discount current value
            $packaging_fee = get_post_meta( $post->ID, 'wcb2b_group_packaging_fee', true );
            $packaging_fee_type = isset( $packaging_fee['type'] ) ? $packaging_fee['type'] : '';
            $packaging_fee_value = isset( $packaging_fee['value'] ) ? $packaging_fee['value'] : '';

            // Display fields
            $options = '<option value="percent" ' . selected( $packaging_fee_type, 'percent', false ) . '>' . esc_html__( 'Percentage', 'woocommerce-b2b' ) . '</option>' .
                        '<option value="amount" ' . selected( $packaging_fee_type, 'amount', false ) . '>' . esc_html__( 'Amount', 'woocommerce-b2b' ) . '</option>';

            printf( '<p><label for="wcb2b_packaging_fee_type">%s</label><br /><select name="wcb2b_group_packaging_fee[type]">%s</select></p>',
                __( 'Fee type', 'woocommerce-b2b' ),
                $options
            );
            printf( '<p><label for="wcb2b_packaging_fee_value">%s</label><br /><input type="text" name="wcb2b_group_packaging_fee[value]" value="%s" pattern="[0-9]+([\.][0-9]+)?" title="' . __( 'This should be a number with up to 2 decimal places, with (.) as decimal separator', 'woocommerce-b2b' ) . '" /></p>',
                __( 'Fee amount', 'woocommerce-b2b' ),
                $packaging_fee_value
            );
        }, 'wcb2b_group', 'side', 'low' );

        // Min purchase amount meta box
        add_meta_box( 'wcb2b_group-min_purchase_amount-meta_box', __( 'Min purchase amount', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-min_purchase_amount-nonce' );

            // Retrieve group discount current value
            $group_min_purchase_amount = get_post_meta( $post->ID, 'wcb2b_group_min_purchase_amount', true );

            // Display fields
            printf( '<p><input type="number" name="wcb2b_group_min_purchase_amount" value="%s" step="1" min="0" title="' . __( 'To disable for this group, set to 0', 'woocommerce-b2b' ) . '" /></p>',
                $group_min_purchase_amount
            );
        }, 'wcb2b_group', 'side', 'low' );

        // Tax exemption meta box
        add_meta_box( 'wcb2b_group-tax_exemption-meta_box', __( 'Tax exemption', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-tax_exemption-nonce' );

            // Retrieve group discount current value
            $tax_exemption = get_post_meta( $post->ID, 'wcb2b_group_tax_exemption', true );

            // Display fields
            printf( '<p><input type="checkbox" name="wcb2b_group_tax_exemption" value="1" %s>' . __( 'Enable tax exemption for this group', 'woocommerce-b2b' ) . '</p>',
                checked( $tax_exemption, true, false )
            );
        }, 'wcb2b_group', 'side', 'low' );

        if ( get_option( 'wcb2b_tax_display' ) === 'yes' ) {

            // Tax display meta box
            add_meta_box( 'wcb2b_group-tax_display-meta_box', __( 'Display prices', 'woocommerce-b2b' ), function( $post ) {
                // Make sure the form request comes from WordPress
                wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-tax_display-nonce' );

                // Retrieve group tax display current value
                $tax_display = get_post_meta( $post->ID, 'wcb2b_group_tax_display', true );

                // Build options
                $options_html = '';
                $options = array(
                    0 => __( '-- As globally defined --', 'woocommerce-b2b' ),
                    'incl' => __( 'Including tax', 'woocommerce-b2b' ),
                    'excl' => __( 'Excluding tax', 'woocommerce-b2b' ),
                );
                foreach ( $options as $key => $value ) {
                    $options_html .= sprintf( '<option value="%s" %s>%s</option>',
                        $key,
                        selected( $key, $tax_display, false ),
                        $value
                    );
                }
                // Display fields
                printf( '<p><select name="wcb2b_group_tax_display">%s</select></p>',
                    $options_html
                );
            }, 'wcb2b_group', 'side', 'low' );

            // Price suffix meta box
            add_meta_box( 'wcb2b_group-price_suffix-meta_box', __( 'Price suffix', 'woocommerce-b2b' ), function( $post ) {
                // Make sure the form request comes from WordPress
                wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-price_suffix-nonce' );

                // Retrieve price suffix current value
                $price_suffix = get_post_meta( $post->ID, 'wcb2b_group_price_suffix', true );

                // Display fields
                printf( '<p><input type="text" name="wcb2b_group_price_suffix" value="%s" /></p>',
                    $price_suffix
                );
                printf( '<span>%s</span>', __( 'Define text to show after your product prices. This could be, for example, "inc. Vat" to explain your pricing. You can also have prices substituted here using one of the following: {price_including_tax}, {price_excluding_tax}.', 'woocommerce-b2b' ) );
            }, 'wcb2b_group', 'side', 'low' );
        }

        if ( get_option( 'wcb2b_moderate_customer_registration' ) === 'yes' ) {

            // Tax display meta box
            add_meta_box( 'wcb2b_group-moderate_registration-meta_box', __( 'Moderate registration', 'woocommerce-b2b' ), function( $post ) {
                // Make sure the form request comes from WordPress
                wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-moderate_registration-nonce' );

                // Retrieve group moderate registration current value
                $moderate_registration = get_post_meta( $post->ID, 'wcb2b_group_moderate_registration', true );

                // Display fields
                printf( '<p><input type="checkbox" name="wcb2b_group_moderate_registration" value="1" %s>' . __( 'Enable login only after admin approval for customers in this group', 'woocommerce-b2b' ) . '</p>',
                    checked( $moderate_registration, true, false )
                );
            }, 'wcb2b_group', 'side', 'low' );
        }

        // Payment methods meta box
        add_meta_box( 'wcb2b_group-gateways-meta_box', __( 'Disable payment methods for this group', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-gateways-nonce' );

            if ( $gateways = WC()->payment_gateways->payment_gateways() ) {
                // Retrieve group allowed gateways value
                $group_gateways = get_post_meta( $post->ID, 'wcb2b_group_gateways', true );
                if ( empty( $group_gateways ) ) {
                    $group_gateways = array();
                }
                foreach ( $gateways as $gateway ) {
                    if ( 'yes' == $gateway->enabled ) {
                        // Display fields
                        printf( '<p><input type="checkbox" name="wcb2b_group_gateways[]" value="%s" %s /> %s</p>',
                            $gateway->id,
                            is_array( $group_gateways ) ? checked( in_array( $gateway->id, $group_gateways ), true, false ) : false,
                            $gateway->title
                        );
                    }
                }
            }
        }, 'wcb2b_group', 'normal', 'low' );

        // Shipping methods meta box
        add_meta_box( 'wcb2b_group-shippings-meta_box', __( 'Disable shipping methods for this group', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-shippings-nonce' );

            $shipping_rates = array();
            $zones = WC_Shipping_Zones::get_zones();
            foreach ( $zones as $zone ) {
                foreach ( $zone['shipping_methods'] as $zone_shipping_methods ) {
                    $shipping_rates[$zone_shipping_methods->id][] = array(
                        'title' => $zone_shipping_methods->title,
                        'id'    => sprintf( '%s:%s', $zone_shipping_methods->id, $zone_shipping_methods->instance_id ),
                        'zone'  => $zone['zone_name']
                    );
                }
            }

            if ( $shipping_methods = WC()->shipping->get_shipping_methods() ) {
                // Retrieve group allowed gateways value
                $group_shippings = get_post_meta( $post->ID, 'wcb2b_group_shippings', true );
                if ( empty( $group_shippings ) ) {
                    $group_shippings = array();
                }
                foreach ( $shipping_methods as $shipping_method_key => $shipping_method ) {
                    // Display fields
                    $shipping_method_field = sprintf( '<p class="wcb2b_shipping_method"><input type="checkbox" name="wcb2b_group_shippings[]" value="%s" %s /> %s</p>',
                        $shipping_method->id,
                        is_array( $group_shippings ) ? checked( in_array( $shipping_method->id, $group_shippings ), true, false ) : false,
                        $shipping_method->method_title
                    );
                    echo apply_filters_deprecated( 'wcb2b_admin_shipping_methods', array( $shipping_method_field, $shipping_method_key, $shipping_method, $group_shippings), '3.0.6' );

                    if ( isset( $shipping_rates[$shipping_method->id] ) ) {
                        foreach ( $shipping_rates[$shipping_method->id] as $shipping_rate_key => $shipping_rate ) {
                            printf( '<p class="wcb2b_shipping_rate" data-parent="%s"><input type="checkbox" name="wcb2b_group_shippings[]" class="%s" value="%s" %s /> %s <em>%s</em></p>',
                                $shipping_method->id,
                                is_array( $group_shippings ) && in_array( $shipping_method->id, $group_shippings ) ? 'disabled' : false,
                                $shipping_rate['id'],
                                is_array( $group_shippings ) ? checked( in_array( $shipping_rate['id'], $group_shippings ), true, false ) : false,
                                $shipping_rate['title'],
                                sprintf( __( 'Defined in "%s" zone', 'woocommerce-b2b' ), $shipping_rate['zone'] )
                            );
                        }
                    }
                }
            }
        }, 'wcb2b_group', 'normal', 'low' );

        if ( get_option( 'wcb2b_product_cat_visibility' ) === 'yes' ) {

            // Product category meta box
            add_meta_box( 'wcb2b_group-taxonomy-meta_box', __( 'Product categories visibility for this group', 'woocommerce-b2b' ), function( $post ) {
                if ( $terms = get_terms( array( 'taxonomy' => 'product_cat', 'hide_empty' => false ) ) ) {
                    foreach ( $terms as $term ) {
                        $group_visibility = get_term_meta( $term->term_id, 'wcb2b_group_visibility', true );
                        $is_visibile = is_array( $group_visibility ) ? in_array( $post->ID, $group_visibility ) : false;

                        // Display fields
                        printf( '<p><a href="%s"><span class="dashicons dashicons-external"></span></a> %s</p>',
                            get_edit_term_link( $term->term_id, 'product_cat' ),
                            ( ! $is_visibile ? '<s>' : false ) . $term->name . ( ! $is_visibile ? '</s>' : false )
                        );
                    }
                }
            }, 'wcb2b_group', 'normal', 'low' );

        }

        // Terms and conditions meta box
        add_meta_box( 'wcb2b_group-terms_conditions-meta_box', __( 'Terms and conditions for this group', 'woocommerce-b2b' ), function( $post ) {
            // Make sure the form request comes from WordPress
            wp_nonce_field( basename( __FILE__ ), 'wcb2b_group-terms_conditions-nonce' );

            $terms_conditions = get_post_meta( $post->ID, 'wcb2b_group_terms_conditions', true );
            wp_editor(
                $terms_conditions,
                'wcb2b_group_terms_conditions',
                array(
                    'textarea_name' => 'wcb2b_group_terms_conditions',
                    'media_buttons' => false,
                    'tinymce'       => apply_filters( 'wcb2b_group_terms_conditions_editor', array(
                        'toolbar1'      => 'bold,italic,underline,separator,alignleft,aligncenter,alignright,separator,link,unlink,undo,redo',
                        'toolbar2'      => '',
                        'toolbar3'      => '',
                    ) )
                )
            );
            printf( '<br /><p class="description">%s</p>',
                sprintf( __( 'You can add this content by using the shortcode %s', 'woocommerce-b2b' ),
                    '<strong>[wcb2bgrouptermsconditions]</strong>'
                )
            );
        }, 'wcb2b_group', 'normal', 'low' );

        // Shortcode meta box
        add_meta_box( 'wcb2b_group-shortcode-meta_box', __( 'Shortcodes tips', 'woocommerce-b2b' ), function( $post ) {
            printf( '<br /><p class="description">%s</p>',
                sprintf( __( 'Allow content visibility only to this group: %s', 'woocommerce-b2b' ),
                    '<strong>[wcb2brestrictedcontent allowed="' . $post->ID . '"][/wcb2brestrictedcontent]</strong>'
                )
            );
        }, 'wcb2b_group', 'normal', 'low' );
    }

    /**
     * Store custom field meta box data
     * 
     * @param int $post_id The current post ID
     */
    public function save_meta_box_fields( $post_id ) {
        // Only ShopManager can edit customer group
        if ( current_user_can( 'manage_woocommerce' ) ) {

            // Check bulk edit nonce
            if ( wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-posts' ) ) {
                // Store group tax exemption flag
                $tax_exemption = $_REQUEST['wcb2b_group_tax_exemption-bulk'] ?? -1;
                if ( '-1' !== $tax_exemption ) {
                    update_post_meta( $post_id, 'wcb2b_group_tax_exemption', intval( $tax_exemption ) );
                }

                // Store tax display
                $tax_display = $_REQUEST['wcb2b_group_tax_display-bulk'] ?? -1;
                if ( '-1' !== $tax_display ) {
                    update_post_meta( $post_id, 'wcb2b_group_tax_display', wp_kses_post( $tax_display ) );
                }

                // Store moderate registration
                $moderate_registration = $_REQUEST['wcb2b_group_moderate_registration-bulk'] ?? -1;
                if ( '-1' !== $moderate_registration ) {
                    update_post_meta( $post_id, 'wcb2b_group_moderate_registration', intval( $moderate_registration ) );
                }
            }

            // Verify meta box nonce
            if ( ! isset( $_POST['wcb2b_group-discount-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-discount-nonce'], basename( __FILE__ ) ) ) { return; }
            if ( ! isset( $_POST['wcb2b_group-packaging_fee-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-packaging_fee-nonce'], basename( __FILE__ ) ) ) { return; }
            if ( ! isset( $_POST['wcb2b_group-min_purchase_amount-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-min_purchase_amount-nonce'], basename( __FILE__ ) ) ) { return; }
            if ( ! isset( $_POST['wcb2b_group-tax_exemption-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-tax_exemption-nonce'], basename( __FILE__ ) ) ) { return; }
            if ( ! isset( $_POST['wcb2b_group-gateways-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-gateways-nonce'], basename( __FILE__ ) ) ) { return; }
            if ( ! isset( $_POST['wcb2b_group-shippings-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-shippings-nonce'], basename( __FILE__ ) ) ) { return; }
            if ( ! isset( $_POST['wcb2b_group-terms_conditions-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-terms_conditions-nonce'], basename( __FILE__ ) ) ) { return; }
            
            if ( get_option( 'wcb2b_tax_display' ) === 'yes' ) {
                if ( ! isset( $_POST['wcb2b_group-tax_display-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-tax_display-nonce'], basename( __FILE__ ) ) ) { return; }
                if ( ! isset( $_POST['wcb2b_group-price_suffix-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-price_suffix-nonce'], basename( __FILE__ ) ) ) { return; }
            }
            if ( get_option( 'wcb2b_moderate_customer_registration' ) === 'yes' ) {
                if ( ! isset( $_POST['wcb2b_group-moderate_registration-nonce'] ) || ! wp_verify_nonce( $_POST['wcb2b_group-moderate_registration-nonce'], basename( __FILE__ ) ) ) { return; }
            }
            
            // Return if autosave
            if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) { return; }

            // Store group discount
            $group_discount = isset( $_POST['wcb2b_group_discount'] ) ? $_POST['wcb2b_group_discount'] : '';
            update_post_meta( $post_id, 'wcb2b_group_discount', wc_format_decimal( $group_discount ) );

            // Store group packaging fee
            $group_packaging_fee = isset( $_POST['wcb2b_group_packaging_fee'] ) ? $_POST['wcb2b_group_packaging_fee'] : '';
            update_post_meta( $post_id, 'wcb2b_group_packaging_fee', $group_packaging_fee );

            // Store group min purchase amount
            $group_min_purchase_amount = isset( $_POST['wcb2b_group_min_purchase_amount'] ) ? $_POST['wcb2b_group_min_purchase_amount'] : 0;
            update_post_meta( $post_id, 'wcb2b_group_min_purchase_amount', intval( $group_min_purchase_amount ) );

            // Store group tax exemption flag
            $tax_exemption = isset( $_POST['wcb2b_group_tax_exemption'] ) ? $_POST['wcb2b_group_tax_exemption'] : 0;
            update_post_meta( $post_id, 'wcb2b_group_tax_exemption', intval( $tax_exemption ) );

            // Store tax display
            $tax_display = isset( $_POST['wcb2b_group_tax_display'] ) ? wp_kses_post( $_POST['wcb2b_group_tax_display'] ) : '';
            update_post_meta( $post_id, 'wcb2b_group_tax_display', $tax_display );

            // Store price suffix
            $price_suffix = isset( $_POST['wcb2b_group_price_suffix'] ) ? wp_kses_post( $_POST['wcb2b_group_price_suffix'] ) : '';
            update_post_meta( $post_id, 'wcb2b_group_price_suffix', $price_suffix );

            // Store moderate registration
            $moderate_registration = isset( $_POST['wcb2b_group_moderate_registration'] ) ? $_POST['wcb2b_group_moderate_registration'] : 0;
            update_post_meta( $post_id, 'wcb2b_group_moderate_registration', intval( $moderate_registration ) );

            // Store group gateways
            $group_gateways = isset( $_POST['wcb2b_group_gateways'] ) ? $_POST['wcb2b_group_gateways'] : '';
            update_post_meta( $post_id, 'wcb2b_group_gateways', $group_gateways );

            // Store group shippings
            $group_shippings = isset( $_POST['wcb2b_group_shippings'] ) ? $_POST['wcb2b_group_shippings'] : '';
            update_post_meta( $post_id, 'wcb2b_group_shippings', $group_shippings );

            // Store group shippings
            $group_terms_conditions = isset( $_POST['wcb2b_group_terms_conditions'] ) ? wp_kses_post( $_POST['wcb2b_group_terms_conditions'] ) : '';
            update_post_meta( $post_id, 'wcb2b_group_terms_conditions', $group_terms_conditions );

            // To fix price caches
            wc_delete_product_transients();
        }
    }

    /**
     * Add new columns in groups list
     * 
     * @param  array $columns Array of column name => label
     * @return array
     */
    public function render_columns( $columns ) {
        return  array_slice( $columns, 0, -1, true ) + 
                array( 'wcb2b_discount' => __( 'Discount', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_min_purchase_amount' => __( 'Min purchase amount', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_packaging_fee' => __( 'Packaging fee', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_payment_methods' => __( 'Allowed payment methods', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_shipping_methods' => __( 'Allowed shipping methods', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_tax_display' => __( 'Display prices', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_tax_exemption' => __( 'Tax exemption', 'woocommerce-b2b' ) ) +
                array( 'wcb2b_moderate_registration' => __( 'Moderate registration', 'woocommerce-b2b' ) ) +
                array_slice( $columns, -1, null, true );
    }

    /**
     * Retrieve column value in groups list
     *
     * @param string $column Column name
     * @param int $post_id ID of the currently-listed group
     */
    public function render_columns_value( $column, $post_id ) {
        if ( $column === 'wcb2b_discount' ) {
            if ( $discount = get_post_meta( $post_id, 'wcb2b_group_discount', true ) ) {
                printf( '%s%%', $discount );
            }
        }
        if ( $column === 'wcb2b_min_purchase_amount' ) {
            echo get_post_meta( $post_id, 'wcb2b_group_min_purchase_amount', true );
        }
        if ( $column === 'wcb2b_packaging_fee' ) {
            if ( $fee = get_post_meta( $post_id, 'wcb2b_group_packaging_fee', true ) ) {
                printf( '%s%s',
                    $fee['value'],
                    $fee['type'] == 'percent' ? '%' : false
                );
            }
        }
        if ( $column === 'wcb2b_payment_methods' ) {
            if ( $gateways = WC()->payment_gateways->payment_gateways() ) {
                // Retrieve group allowed gateways value
                $group_gateways = get_post_meta( $post_id, 'wcb2b_group_gateways', true );
                if ( empty( $group_gateways ) ) {
                    $group_gateways = array();
                }
                foreach ( $gateways as $key => $gateway ) {
                    if ( ! in_array( $gateway->id, $group_gateways ) && 'yes' == $gateway->enabled ) {
                        printf( '<div>%s</div>', $gateway->title );
                    }
                }
            }
        }
        if ( $column === 'wcb2b_shipping_methods' ) {
            if ( $shippings = WC()->shipping->get_shipping_methods() ) {
                // Retrieve group allowed gateways value
                $group_shippings = get_post_meta( $post_id, 'wcb2b_group_shippings', true );
                if ( empty( $group_shippings ) ) {
                    $group_shippings = array();
                }
                foreach ( $shippings as $key => $shipping ) {
                    if ( ! in_array( $shipping->id, $group_shippings ) ) {
                        printf( '<div>%s</div>', $shipping->method_title );
                    }
                }
            }
        }
        if ( $column === 'wcb2b_tax_display' ) {
            $value = get_post_meta( $post_id, 'wcb2b_group_tax_display', true );
            switch ( $value ) {
                default :
                    esc_html_e( '-- As globally defined --', 'woocommerce-b2b' );
                    break;
                case 'incl' :
                    esc_html_e( 'Including tax', 'woocommerce-b2b' );
                    break;
                case 'excl' :
                    esc_html_e( 'Excluding tax', 'woocommerce-b2b' );
                    break;
            }
        }
        if ( $column === 'wcb2b_tax_exemption' ) {
            printf('<span class="dashicons dashicons-%s"></span>',
                get_post_meta( $post_id, 'wcb2b_group_tax_exemption', true ) ? 'yes' : 'no'
            );
        }
        if ( $column === 'wcb2b_moderate_registration' ) {
            printf('<span class="dashicons dashicons-%s"></span>',
                get_post_meta( $post_id, 'wcb2b_group_moderate_registration', true ) ? 'yes' : 'no'
            );
        }
    }

    /**
     * Make column sortable in groups list
     * 
     * @param  array $columns Array of column name => label
     * @return array
     */
    public function make_columns_sortable( $columns ) {
        $columns['wcb2b_discount'] = 'wcb2b_discount';
        $columns['wcb2b_min_purchase_amount'] = 'wcb2b_group_min_purchase_amount';
        $columns['wcb2b_packaging_fee'] = 'wcb2b_packaging_fee';
        $columns['wcb2b_payment_methods'] = 'wcb2b_payment_methods';
        $columns['wcb2b_shipping_methods'] = 'wcb2b_shipping_methods';
        return $columns;
    }

    /**
     * Default hidden columns in groups 
     *   
     * @param array $hidden Hidden columns
     * @param object $screen Current screen instance
     * @return array
     */
    public function default_hidden_columns( $hidden, $screen ) {
        if ( 'wcb2b_group' == $screen->post_type ) {
            $hidden[] = 'wcb2b_tax_display';
            $hidden[] = 'wcb2b_tax_exemption';
            $hidden[] = 'wcb2b_moderate_registration';
            $hidden[] = 'date';
        }
        return $hidden;
    }

    /**
     * Add fields for quick bulk edit
     * 
     * @param string $column_name Current column slug
     * @param string $post type Current post type
     */
    public function quick_edit_fields( $column_name, $post_type ) {
        switch( $column_name ) {
            case 'wcb2b_tax_display' :
                $options_html = '';
                foreach ( array(
                    '-1'   => __( '-- No Change --', 'woocommerce-b2b' ),
                    '0'    => __( '-- As globally defined --', 'woocommerce-b2b' ),
                    'incl' => __( 'Including tax', 'woocommerce-b2b' ),
                    'excl' => __( 'Excluding tax', 'woocommerce-b2b' ),
                ) as $key => $value ) {
                    $options_html .= sprintf( '<option value="%s">%s</option>',
                        $key,
                        $value
                    );
                }
                printf( '<fieldset class="inline-edit-col-left"><div class="inline-edit-col"><label>%s%s</label></div></fieldset>',
                    sprintf( '<span class="title">%s</span>',
                        esc_html__( 'Display prices', 'woocommerce-b2b' )
                    ),
                    sprintf( '<select name="wcb2b_group_tax_display-bulk">%s</select>',
                        $options_html
                    )
                );
                break;
            case 'wcb2b_tax_exemption' :
                $options_html = '';
                foreach ( array(
                    '-1' => __( '-- No Change --', 'woocommerce-b2b' ),
                    '1'  => __( 'Yes', 'woocommerce-b2b' ),
                    '0'  => __( 'No', 'woocommerce-b2b' ),
                ) as $key => $value ) {
                    $options_html .= sprintf( '<option value="%s">%s</option>',
                        $key,
                        $value
                    );
                }
                printf( '<fieldset class="inline-edit-col-left"><div class="inline-edit-col"><label>%s%s</label></div></fieldset>',
                    sprintf( '<span class="title">%s</span>',
                        esc_html__( 'Tax exemption', 'woocommerce-b2b' )
                    ),
                    sprintf( '<select name="wcb2b_group_tax_exemption-bulk">%s</select>',
                        $options_html
                    )
                );
                break;
            case 'wcb2b_moderate_registration' :
                $options_html = '';
                foreach ( array(
                    '-1' => __( '-- No Change --', 'woocommerce-b2b' ),
                    '1'  => __( 'Yes', 'woocommerce-b2b' ),
                    '0'  => __( 'No', 'woocommerce-b2b' ),
                ) as $key => $value ) {
                    $options_html .= sprintf( '<option value="%s">%s</option>',
                        $key,
                        $value
                    );
                }
                printf( '<fieldset class="inline-edit-col-left"><div class="inline-edit-col"><label>%s%s</label></div></fieldset>',
                    sprintf( '<span class="title">%s</span>',
                        esc_html__( 'Moderate registration', 'woocommerce-b2b' )
                    ),
                    sprintf( '<select name="wcb2b_group_moderate_registration-bulk">%s</select>',
                        $options_html
                    )
                );
                break;
        }
    }

    /**
     * Remove guest group action delete
     * 
     * @param array $actions Actions list
     * @param object $post Current listed post instance
     * @return array
     */
    public function remove_guest_group_delete( $actions, $post ) {
        if ( $post->ID == get_option( 'wcb2b_guest_group' ) ) {
            unset( $actions['trash'] );
            unset( $actions['delete'] );
        }
        return $actions;
    }

    /**
     * Prevent guest group action delete
     *
     * @param array $allcaps All capabilities
     * @param array $caps User capabilities
     * @param array $args Arguments
     * @return array
     */
    public function prevent_guest_group_delete( $allcaps, $caps, $args ) {
        $guest_group_id = get_option( 'wcb2b_guest_group' );
        if ( isset( $args[0] ) && isset( $args[2] ) && $args[2] == $guest_group_id && $args[0] == 'delete_post' ) {
            $allcaps[ $caps[0] ] = false;
        }
        return $allcaps;
    }

    /**
     * Create visibility term meta if not exists 
     * 
     * @param string $option Current option name
     * @param mixed $old_value Previous option value
     * @param mixed $value New option value
     */
    public function updated_group( $option, $old_value, $value ) {
        if ( $option === 'wcb2b_product_cat_visibility' ) {
            if ( $value === 'yes' ) {
                if ( $terms = get_terms( array( 'taxonomy' => 'product_cat', 'hide_empty' => false ) ) ) {
                    $group_ids = wp_list_pluck( wcb2b_get_groups()->posts, 'ID' );
                    $group_ids = array_map( 'intval', array_filter( array_unique( $group_ids ) ) );

                    foreach ( $terms as $term ) {
                        add_term_meta( $term->term_id, 'wcb2b_group_visibility', $group_ids, true );
                    }
                }
            }
        }
        if ( $option === 'wcb2b_page_visibility' ) {
            if ( $value === 'yes' ) {
                if ( $pages = get_pages() ) {
                    $group_ids = wp_list_pluck( wcb2b_get_groups()->posts, 'ID' );
                    $group_ids = array_map( 'intval', array_filter( array_unique( $group_ids ) ) );

                    foreach ( $pages as $page ) {
                        add_post_meta( $page->ID, 'wcb2b_group_visibility', $group_ids, true );
                    }
                }
            }
        }
    }

    /**
     * On save group (only add new) add to all terms visibility
     * 
     * @param string $new_status New group status
     * @param string $old_status Previous group status
     * @param object $post Current group instance
     */
    public function updated_group_status( $new_status, $old_status, $post ) {
        if ( $post->post_type == 'wcb2b_group' ) {
            if ( $old_status == 'new' ) {
                if ( $terms = get_terms( array( 'taxonomy' => 'product_cat', 'hide_empty' => false ) ) ) {
                    // Update visible groups
                    foreach ( $terms as $term ) {
                        if ( ! $visibility = get_term_meta( $term->term_id, 'wcb2b_group_visibility', true ) ) {
                            // Fix if term meta is empty
                            $visibility = array();
                        }
                        array_push( $visibility, $post->ID );
                        $visibility = array_filter( array_unique( $visibility ) );

                        // Add created group id to term meta
                        update_term_meta( $term->term_id, 'wcb2b_group_visibility', $visibility );
                    }
                }
                if ( $pages = get_pages() ) {
                    foreach ( $pages as $page ) {
                        if ( ! $visibility = get_post_meta( $page->ID, 'wcb2b_group_visibility', true ) ) {
                            // Fix if term meta is empty
                            $visibility = array();
                        }
                        array_push( $visibility, $post->ID );

                        // Add created group id to term meta
                        update_post_meta( $page->ID, 'wcb2b_group_visibility', $visibility );
                    }
                }
            }
        }
    }

    /**
     * On delete group, remove visibility from all terms and assign default group to customers
     * 
     * @param int $post_id Current group ID
     */
    public function deleted_group( $post_id ) {
        if ( $terms = get_terms( array( 'taxonomy' => 'product_cat', 'hide_empty' => false ) ) ) {
            foreach ( $terms as $term ) {
                if ( ! $visibility = get_term_meta( $term->term_id, 'wcb2b_group_visibility', wcb2b_get_groups()->posts, true ) ) {
                    // Fix if term meta is empty
                    $visibility = array();
                }
                if ( $key = array_search( $post_id, (array)$visibility ) !== false ) {
                    unset( $visibility[$key] );
                }
                $visibility = array_filter( array_unique( $visibility ) );

                // Update groups id to term meta
                update_term_meta( $term->term_id, 'wcb2b_group_visibility', $visibility );
            }
        }
        if ( $pages = get_pages() ) {
            foreach ( $pages as $page ) {
                if ( ! $visibility = get_post_meta( $page->ID, 'wcb2b_group_visibility', wcb2b_get_groups()->posts, true ) ) {
                    // Fix if term meta is empty
                    $visibility = array();
                }
                if ( $key = array_search( $post_id, (array)$visibility ) !== false ) {
                    unset( $visibility[$key] );
                }
                $visibility = array_filter( array_unique( $visibility ) );

                // Update groups id to term meta
                update_post_meta( $page->ID, 'wcb2b_group_visibility', $visibility );
            }
        }

        // Get all customers belonging deleted group
        $query = new WP_User_Query( array(
            'role__in'   => apply_filters( 'wcb2b_has_role_customer', array( 'customer' ) ),
            'meta_key'   => 'wcb2b_group',
            'meta_value' => $post_id,
            'fields'     => 'ID'
        ) );
        $customers = $query->get_results();

        // Get the default group
        $default_group = get_option( 'wcb2b_default_group', false );
        
        // Remove group from default option, if exists
        if ( $default_group == $post_id ) {
            update_option( 'wcb2b_default_group', get_option( 'wcb2b_guest_group' ) );
        }
        
        // Set default group to all customers belonging deleted group
        if ( ! empty( $customers ) ) {
            foreach ( $customers as $customer ) {
                update_user_meta( $customer, 'wcb2b_group', $default_group ); 
            }
        }

        // To fix price caches
        wc_delete_product_transients();
    }

}

return new WCB2B_Post_Types();