<?php

/**
 * WooCommerce B2B REST API support
 *
 * @version 3.1.4
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_Rest_Controller Class
 */
class WCB2B_Rest_Controller {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Init current class hooks
     */
    public function init_hooks() {
        add_action( 'rest_api_init', array( $this, 'add_group_controller' ), 10 );
        add_action( 'rest_api_init', array( $this, 'add_product_cat_meta' ) );
        add_filter( 'woocommerce_rest_prepare_customer', array( $this, 'add_customer_meta' ), 90, 2 );
        add_filter( 'woocommerce_rest_prepare_shop_order_object', array( $this, 'add_order_meta' ), 90, 2 );
        add_filter( 'woocommerce_rest_prepare_product_object', array( $this, 'add_product_meta' ), 90, 2 );
        add_filter( 'woocommerce_rest_prepare_product_variation_object', array( $this, 'add_product_meta' ), 90, 2 );
    }

    /**
     * Add new REST API class
     */
    public function add_group_controller() {
        include_once WCB2B_ABSPATH . 'includes/legacy/api/class-wcb2b-api-groups.php';
        $controller = new WCB2B_API_Groups();
        $controller->register_routes();
    }

    /**
     * Add support for term meta to REST API
     */
    public function add_product_cat_meta() {
        register_rest_field( 'product_cat',
            'meta', array(
                'get_callback'    => function( $object, $field, $request ) {
                    return get_term_meta( $object['id'] );
                },
                'update_callback' => function( $value, $object, $field ){
                    return update_post_meta( $object['id'], $field, $value );
                },
                'schema'          => null,
            )
        );
    }

    /**
     * Add customer fields to REST API
     *
     * @param object $response REST API response object
     * @param object $customer Current customer object
     * @return object
     */
    public function add_customer_meta( $response, $customer ) {
        // Invoice email
        $response->data['billing']['invoice_email'] = $response->data['wcb2b_invoice_email'] = get_user_meta( $customer->ID, 'billing_invoice_email', true );

        // VAT number
        $response->data['billing']['vat_number'] = $response->data['wcb2b_vat_number'] = get_user_meta( $customer->ID, 'billing_vat', true );

        // Status
        $response->data['wcb2b_status'] = (int)get_user_meta( $customer->ID, 'wcb2b_status', true );

        // Upnaid orders limit
        $response->data['wcb2b_unpaid_limit'] = (int)get_user_meta( $customer->ID, 'wcb2b_unpaid_limit', true );

        // Group
        $response->data['wcb2b_group'] = array();
        if ( $group = get_post( get_user_meta( $customer->ID, 'wcb2b_group', true ) ) ) {
            $response->data['wcb2b_group']['id'] = $group->ID;
            $response->data['wcb2b_group']['name'] = $group->post_title;
            $response->data['wcb2b_group']['discount'] = get_post_meta( $group->ID, 'wcb2b_group_discount', true );
        }
        return $response;
    }

    /**
     * Add order fields to REST API
     *
     * @param object $response REST API response object
     * @param object $order Current order object
     * @return object
     */
    public function add_order_meta( $response, $order ) {
        // External invoice number
        $response->data['wcb2b_external_invoice_number'] = (int)$order->get_meta( '_wcb2b_external_invoice_number', true );
        // Group
        $response->data['wcb2b_group'] = (int)$order->get_meta( '_wcb2b_group', true );
        // Total weight
        $response->data['wcb2b_total_weight'] = (int)$order->get_meta( '_total_weight', true );
        return $response;
    }

    /**
     * Add product fields to REST API
     *
     * @param object $response REST API response object
     * @param object $customer Current product object
     * @return object
     */
    public function add_product_meta( $response, $product ) {
        // Group prices
        $response->data['wcb2b_group_prices'] = get_post_meta( $product->get_ID(), 'wcb2b_product_group_prices' );

        // Group tier prices
        $response->data['wcb2b_group_tier_prices'] = get_post_meta( $product->get_ID(), 'wcb2b_product_group_tier_prices' );

        // Barcode
        $response->data['wcb2b_barcode'] = get_post_meta( $product->get_ID(), 'wcb2b_barcode' );

        // Package quantity
        $response->data['wcb2b_package_quantity'] = get_post_meta( $product->get_ID(), 'wcb2b_product_group_packages' );

        // Minimum quantity
        $response->data['wcb2b_min_quantity'] = get_post_meta( $product->get_ID(), 'wcb2b_product_group_min' );

        // Maximum quantity
        $response->data['wcb2b_max_quantity'] = get_post_meta( $product->get_ID(), 'wcb2b_product_group_max' );

        // Hidden prices
        $response->data['wcb2b_group_hide_prices'] = get_post_meta( $product->get_ID(), 'wcb2b_product_group_hide_prices' );
        return $response;
    }

}

return new WCB2B_Rest_Controller();