<?php

/**
 * WooCommerce B2B Settings
 *
 * @version 3.1.4
 */

defined( 'ABSPATH' ) || exit;

if ( class_exists( 'WCB2B_Settings', false ) ) {
    return new WCB2B_Settings();
}

/**
 * WCB2B_Settings
 */
class WCB2B_Settings extends WC_Settings_Page {

    /**
     * Constructor
     */
    public function __construct() {
        $this->id    = 'woocommerce-b2b';
        $this->label = __( 'B2B', 'woocommerce-b2b' );

        parent::__construct();
    }

    /**
     * Get settings array
     *
     * @return array
     */
    public function get_settings() {
        // If SOAP extension is not available, disable VIES validation
        if ( ! $soap = extension_loaded( 'soap' ) ) {
            update_option( 'wcb2b_vies_validation', 'no' );
        }

        $countries = new WC_Countries();
        $groups = wcb2b_get_groups();

        $settings = apply_filters( 'wcb2b_general_settings', array(
            /* GENERAL SETTINGS */
            array(
                'name'     => __( 'General settings', 'woocommerce-b2b' ),
                'type'     => 'title',
                'desc'     => __( 'This section controls general options to manage your shop.', 'woocommerce-b2b' )
            ),
            array(
                'name'              => __( 'Restricted catalog', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'desc'              => __( 'Restict catalog visibility to logged in customers', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Redirect not logged in users to login page to have a restricted access', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_restricted_catalog'
            ),
            array(
                'name'              => __( 'Quick order page', 'woocommerce-b2b' ),
                'type'              => 'select',
                'default'           => 'null',
                'class'             => 'wc-enhanced-select',
                'desc'              => sprintf( __( 'Page to use for quick orders. Remember to add the shortcode %1$s.', 'woocommerce-b2b' ), '<strong>[wcb2bquickorder]</strong>'),
                'id'                => 'wcb2b_quick_order_page',
                'options'           => array_reduce( get_pages( array( 'post_status' => array( 'publish', 'draft' ) ) ), function( $result, $item ) {
                    $item = (array)$item;
                    $result[$item['ID']] = $item['post_title'];
                    return $result;
                } )
            ),
            array(
                 'type' => 'sectionend'
            ),

            /* PRODUCTS SETTINGS */
            array(
                'name'              => __( 'Products settings', 'woocommerce-b2b' ),
                'type'              => 'title',
                'desc'              => __( 'This section controls product and product categories options to manage prices and visibility.', 'woocommerce-b2b' )
            ),

            // Show prices only to logged in customers 
            array(
                'name'              => __( 'Hide prices', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Hide prices for not logged in customers to all products', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Price is replaced by a message linked to login page.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_hide_prices'
            ),

            // Show RRP prices to B2B customers 
            array(
                'name'              => __( 'RRP', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Show recommended retail price (RRP)', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Show default product regular price as recommended retail price (RRP) to customers belonging WooCommerce B2B groups.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_rrp'
            ),

            // Show barcode
            array(
                'name'              => __( 'Barcode', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Show product barcode', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Display product barcode in product details page.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_barcode'
            ),

            // Prices application rules
            array(
                'name'              => __( 'Price rules', 'woocommerce-b2b' ),
                'type'              => 'radio',
                'default'           => 'global',
                'id'                => 'wcb2b_price_rules',
                'options'           => array(
                    'single' => __( 'Show prices assigned to single product', 'woocommerce-b2b' ),
                    'global' => __( 'Show default prices discounted by customer group percentage', 'woocommerce-b2b' ),
                    'both' => __( 'Show prices assigned to single product discounted by customer group percentage', 'woocommerce-b2b' )
                )
            ),

            // Show "Already bought"
            array(
                'name'              => __( 'Already bought', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Show "Already bought" message', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'If customer has already purchased a product in the past, an "Already bought" message is displayed on product detail page after "Add to cart" button.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_already_bought'
            ),

            // Show product sales
            array(
                'name'              => __( 'Total sales', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Show product total sales', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Show the total number of product sales on the product detail page.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_sales'
            ),

            // Show shippings in product tabs
            array(
                'name'              => __( 'Shippings tab', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Show shippings tab', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Shows additional tab with shippings summary table on product detail page', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_shippings_tab'
            ),

            // Display percentage in user area
            array(
                'name'              => __( 'Show discount', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Customers can view discount amount assigned to them in their own account area', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_customer_discount',
                'checkboxgroup'     => 'start'
            ),

            // Display percentage in product page
            array(
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Customers can view discount amount assigned to them in single product page', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_customer_discount_product',
                'checkboxgroup'     => 'end'
            ),

            // Tax exemption by country
            array(
                'name'              => __( 'Tax exemption by country', 'woocommerce-b2b' ),
                'type'              => 'multiselect',
                'default'           => 'null',
                'class'             => 'wc-enhanced-select',
                'desc'              => __( 'Apply the tax exemption in the shopping cart for customers belonging to a group (not GUEST) with billing country among those chosen.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_tax_exemption_countries',
                'options'           => $countries->get_countries()
            ),

            // Display prices with tax or without tax according to customer group
            array(
                'name'              => __( 'Display prices tax', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Enable prices tax display management by group', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'You can set the display of prices including or excluding taxes in each group page.', 'woocommerce-b2b' )  
                                    . '<br />'
                                    . __( 'When enabled and set, override global options "Display prices in the shop" and "Display prices during cart and checkout" behaviour in "Tax" tab under WooCommerce settings', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_tax_display'
            ),

            // Enable pages by group
            array(
                'name'              => __( 'Pages visibility', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'id'                => 'wcb2b_page_visibility',
                'desc'              => __( 'Display pages management by group', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'When enabled, every page can be set to be visible only to certain groups', 'woocommerce-b2b' )
            ),

            // Enable product categories by group
            array(
                'name'              => __( 'Product categories visibility', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'id'                => 'wcb2b_product_cat_visibility',
                'desc'              => __( 'Display product categories management by group', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'When enabled, every product category can be set to be visible only to certain groups', 'woocommerce-b2b' )
            ),
            
            // Redirect on hidden product categories
            array(
                'name'              => __( 'Redirect on page', 'woocommerce-b2b' ),
                'type'              => 'select',
                'default'           => 'null',
                'class'             => 'wc-enhanced-select',
                'id'                => 'wcb2b_redirect_not_allowed',
                'desc'              => __( 'When users go directly to a not visible product category or a not visible product', 'woocommerce-b2b' ),
                'options'           => array_reduce( get_pages(), function( $result, $item ) {
                    $item = (array)$item;
                    $result[$item['ID']] = $item['post_title'];
                    return $result;
                }, array(
                    'null'  => __( '-- Do nothing --', 'woocommerce-b2b' ),
                    '0'     => __( '404 (not found) page', 'woocommerce-b2b' )
                ) )
            ),
            
            array(
                 'type'             => 'sectionend'
            ),

            /* CART & CHECKOUT SETTINGS */
            array(
                'name'              => __( 'Cart & Checkout settings', 'woocommerce-b2b' ),
                'type'              => 'title',
                'desc'              => __( 'This section controls cart and checkout pages by adding new fields and by setting minimum cart amount to allow to purchase.', 'woocommerce-b2b' )
            ),

            // Add invoice email field in checkout
            array(
                'name'              => __( 'Email address for invoices', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Add an additional field in which customers can communicate email address where they prefer to receive invoices', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Email address for invoices field is added to checkout form, order emails, order details and registration forms (if "REGISTRATION FORMS" option enabled)', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_add_invoice_email',
                'show_if_checked'   => 'option',
                'checkboxgroup'     => 'start'
            ),

            // Make invoice email required
            array(
                'name'              => __( 'Email address for invoices required', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Make email address for invoices field required', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'If disabled, email address for invoices field will be considered as optional', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_invoice_email_required',
                'show_if_checked'   => 'yes', 
                'checkboxgroup'     => true
            ),

            // Add VAT number field in checkout
            array(
                'name'              => __( 'VAT number', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Add VAT number field to billing address', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'VAT number field is added to checkout form, order emails, order details and registration forms (if "REGISTRATION FORMS" option enabled)', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_add_vatnumber',
                'show_if_checked'   => 'option',
                'checkboxgroup'     => 'start'
            ),

            // Make VAT number required
            array(
                'name'              => __( 'VAT number required', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Make VAT number field required', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'If disabled, VAT number field will be considered as optional', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_vatnumber_required',
                'show_if_checked'   => 'yes', 
                'checkboxgroup'     => true
            ),

            // Enable VIES check for EU VAT number
            array(
                'name'              => __( 'VIES validation', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Enable VAT number VIES validation for EU customers', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Validate your EU customers VAT number with VIES WebService', 'woocommerce-b2b' ) . ( $soap ? false : '<br /><span class="error">' . __( 'Attention! To enable VIES check you need PHP SOAP extension is available on your server. Please contact your hosting provider.', 'woocommerce-b2b' ) .'</span>' ),
                'id'                => 'wcb2b_vies_validation',
                'show_if_checked'   => 'yes', 
                'checkboxgroup'     => 'end',
                'disabled'          => ( ! $soap )
            ),

            // Minimum amount
            array(
                'name'              => __( 'Min purchase amount', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Enable minimum purchase amount management by group', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'You can set a dedicated minimum amount to purchase in each group page.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_min_purchase_amount',
                'show_if_checked'   => 'option',
                'checkboxgroup'     => 'start'
            ),

            // Display message in cart
            array(
                'name'              => __( 'Display message in cart', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Display a message in the shopping cart page to alert that minimum amount isn\'t reached yet', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_display_min_purchase_cart_message',
                'show_if_checked'   => 'yes',
                'checkboxgroup'     => ''
            ),

            // Prevent checkout button
            array(
                'name'              => __( 'Prevent proceed to checkout', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Remove "Proceed to checkout" button in shopping cart if customer doesn\'t reach the minimum amount', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_prevent_checkout_button',
                'show_if_checked'   => 'yes', 
                'checkboxgroup'     => 'end'
            ),

            array(
                 'type'             => 'sectionend'
            ),

            /* CUSTOMERS SETTINGS */
            array(
                'name'              => __( 'Customers settings', 'woocommerce-b2b' ),
                'type'              => 'title',
                'desc'              => __( 'This section controls customers options to manage registration forms and account activation.', 'woocommerce-b2b' )
            ),

            // Display unpaid total orders amount
            array(
                'name'              => __( 'Show unpaid amount', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Customers can view unpaid total orders amount in their own account area', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_unpaid_amount'
            ),

            // Display customer group
            array(
                'name'              => __( 'Show customer group', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Customers can view group assigned to them in their own account area', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_show_customer_group'
            ),

            // Moderate registration
            array(
                'name'              => __( 'Moderate registration', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Enable moderate registration management by group', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'You can set admin approval to login in each group page.', 'woocommerce-b2b'),
                'id'                => 'wcb2b_moderate_customer_registration'
            ),

            // Extend registration forms
            array(
                'name'              => __( 'Registration forms', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Add billing data to registration form', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'Extend registration form with billing fields', 'woocommerce-b2b' ) 
                                     . '<br />'
                                     . __( 'To show registration form, you need to enable the option "Allow customers to create an account on the \'My account\' page" in "Accounts & Privacy" tab under WooCommerce settings', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_extend_registration_form'
            ),

            // Registration notice
            array(
                'name'              => __( 'Registration notice', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Send email to admin on every new customer account registration', 'woocommerce-b2b' ),
                'desc_tip'          => __( 'When a customer create an account, an email is send to admin to inform him', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_registration_notice'
            ),

            // Default group assignment
            array(
                'name'              => __( 'Default group', 'woocommerce-b2b' ),
                'type'              => 'select',
                'default'           => '0',
                'class'             => 'wc-enhanced-select',
                'id'                => 'wcb2b_default_group',
                'desc'              => __( 'Assign automatically customer to a group on registration', 'woocommerce-b2b' ),
                'options'           => array_reduce( $groups->posts, function( $result, $item ) {
                    $item = (array)$item;
                    $result[$item['ID']] = $item['post_title'];
                    return $result;
                }, array(
                    '0'      => __( '-- None --', 'woocommerce-b2b' ),
                    'choose' => __( '-- Allow to choose --', 'woocommerce-b2b' )
                ) )
            ),
            
            array(
                 'type'             => 'sectionend'
            ),

            array(
                'name'    => __( 'Endpoints settings', 'woocommerce-b2b' ),
                'type'     => 'title',
                'desc'     => __( 'Endpoints are appended to your page URLs to handle specific actions on the my-account pages. They should be unique.', 'woocommerce-b2b' )
            ),

            // Enable quotations
            array(
                'name'              => __( 'Enable quotations', 'woocommerce-b2b' ),
                'type'              => 'checkbox',
                'default'           => 'no',
                'desc'              => __( 'Enable quotations endpoint in my-account page', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_enable_quotations'
            ),

            array(
                'name'              => __( 'Quotation endpoint', 'woocommerce-b2b' ),
                'type'              => 'text',
                'default'           => 'quotations',
                'desc_tip'          => __( 'Endpoint for the "My account → Quotations" page.', 'woocommerce-b2b' ),
                'id'                => 'wcb2b_account_quotations_endpoint'
            ),

            array(
                'type'    => 'sectionend'
            )
        ) );
        return apply_filters( 'woocommerce_get_settings_' . $this->id, $settings );
    }

    /**
     * Output the settings
     */
    public function output() {
        global $current_section, $hide_save_button;
        $hide_save_button = true;

        $chunks = array(
            'html',
            'admin',
            'settings',
            empty( $current_section ) ? 'general' : $current_section
        );
        $page = implode('-', array_filter($chunks));

        include_once WCB2B_ABSPATH . 'includes/views/settings/html-admin-settings.php';
    }

    /**
     * Save settings
     */
    public function save() {
        $settings = $this->get_settings();
        WC_Admin_Settings::save_fields( $settings );
        flush_rewrite_rules();
    }

}

return new WCB2B_Settings();