<?php

/**
 * WooCommerce B2B REST API group controller
 * 
 * @version 3.0.4
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCB2B_API_Groups Class
 * */
class WCB2B_API_Groups extends WP_REST_Controller
{

    // Properties
    protected $namespace = 'wc/v3';
    protected $base = 'groups';

    /**
     * Register custom routes
     */
    public function register_routes() {
        # GET / POST
        register_rest_route( $this->namespace, '/' . $this->base, array(
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_items' ),
                'permission_callback' => array( $this, 'get_items_permissions_check' ),
                'args'                => array()
            ),
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_item' ),
                'permission_callback' => array( $this, 'create_item_permissions_check' ),
                'args'                => $this->get_endpoint_args_for_item_schema( true ),
            )
        ) );

        # GET / PUT / DELETE
        register_rest_route( $this->namespace, '/' . $this->base . '/(?P<id>[\d]+)', array(
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_item' ),
                'permission_callback' => array( $this, 'get_item_permissions_check' ),
                'args'                => array(
                    'context' => array(
                        'default' => 'view',
                    ),
                ),
            ),
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_item' ),
                'permission_callback' => array( $this, 'update_item_permissions_check' ),
                'args'                => $this->get_endpoint_args_for_item_schema( false ),
            ),
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_item' ),
                'permission_callback' => array( $this, 'delete_item_permissions_check' ),
                'args'                => array(
                    'force' => array(
                        'default' => false,
                    ),
                ),
            )
        ) );

        # SCHEMA
        register_rest_route( $this->namespace, '/' . $this->base . '/schema', array(
            'methods'  => WP_REST_Server::READABLE,
            'callback' => array( $this, 'get_public_item_schema' ),
            'permission_callback' => function() { return true; }
        ) );
    }

    /**
     * Get all group items
     */
    public function get_items( $request ) {
        $data = array();
        $args = array(
            'post_type' => 'wcb2b_group'
        );
        foreach( array(
                    'author',
                    'author_name',
                    'author__in',
                    'author__not_in',
                    'date_query',
                    'exact',
                    'meta_query',
                    'menu_order',
                    'name',
                    'nopaging',
                    'no_found_rows',
                    'offset',
                    'order',
                    'orderby',
                    'page',
                    'paged',
                    'pagename',
                    'perm',
                    'post__in',
                    'post__not_in',
                    'post_status',
                    'posts_per_page',
                    'post_name__in',
                    's',
                    'title'              
            ) as $allowed ) {
            if ( isset( $request[$allowed] ) ) {
                $args[$allowed] = $request[$allowed];
            }
        }
        // Get all groups
        $groups = get_posts( $args );
        if ( $groups ) {
            foreach ( $groups as $group ) {
                $item = $this->prepare_item_for_response( $group, $request );
                $data[] = $this->prepare_response_for_collection( $group );
            }
        }
        return new WP_REST_Response( $data, 200 );
    }

    /**
     * Get single group item
     */
    public function get_item( $request ) {
        // Check ID
        if ( ! isset( $request['id'] ) ) {
            return new WP_Error( 'wcb2b_api_missing_group_id', sprintf( __( 'Missing parameter %s', 'woocommerce-b2b' ), 'id' ), array( 'status' => 400 ) );
        }
        // Get group
        $group = get_post( $request['id'] );
        if ( null === $group ) {
            return new WP_Error( 'wcb2b_api_invalid_group', __( 'Invalid group', 'woocommerce-b2b' ), array( 'status' => 400 ) );
        }
        $data = $this->prepare_item_for_response( $group, $request );
        return new WP_REST_Response( $data, 200 );
    }

    /**
     * Add single group item
     */
    public function create_item( $request ) {
        // Check title
        if ( ! isset( $request['name'] ) ) {
            return new WP_Error( 'wcb2b_api_missing_group_name', sprintf( __( 'Missing parameter %s', 'woocommerce-b2b' ), 'name' ), array( 'status' => 400 ) );
        }
        // Check for duplicate
        if ( get_page_by_title( $request['name'], ARRAY_N, 'wcb2b_group' ) != null ) {
            return new WP_Error( 'wcb2b_api_duplicate_group_name', sprintf( __( 'A group named %s already exists', 'woocommerce-b2b' ), $request['name'] ), array( 'status' => 400 ) );
        }
        $group = $this->prepare_item_for_database( $request );
        // Insert group
        $group_id = wp_insert_post($group);
        if ( ! $group_id ) {
            return new WP_Error( 'wcb2b_api_cannot_create_group', __( 'This group cannot be created', 'woocommerce-b2b' ), array( 'status' => 400 ) );
        }
        return $this->get_item( array( 'id' => $group_id ) );
    }

    /**
     * Update single group item
     */
    public function update_item( $request ) {
        // Check ID
        if ( ! isset( $request['id'] ) ) {
            return new WP_Error( 'wcb2b_api_missing_group_id', sprintf( __( 'Missing parameter %s', 'woocommerce-b2b' ), 'id' ), array( 'status' => 400 ) );
        }
        // Check if group exists
        if ( get_post_status( $request['id'] ) == false ) {
            return new WP_Error( 'wcb2b_api_invalid_group', __( 'Invalid group', 'woocommerce-b2b' ), array( 'status' => 400 ) );
        }

        $group = $this->prepare_item_for_database( $request );
        // Update group and meta
        if ( isset( $request['name'] ) ) {
            $group_id = wp_update_post( array(
                'ID' => $request['id'],
                'post_title' => $request['name']
            ) );
        }
        foreach ( $group['meta_input'] as $meta_key => $meta_value ) {
            update_post_meta( $request['id'], $meta_key, $meta_value );
        }
        return $this->get_item( array( 'id' => $request['id'] ) );
    }

    /**
     *  Delete single group item
     */
    public function delete_item( $request ) {
        // Check ID
        if ( ! isset( $request['id'] ) ) {
            return new WP_Error( 'wcb2b_api_missing_group_id', sprintf( __( 'Missing parameter %s', 'woocommerce-b2b' ), 'id' ), array( 'status' => 400 ) );
        }
        // Check if group exists
        if ( get_post_status( $request['id'] ) == false ) {
            return new WP_Error( 'woocommerce_api_invalid_group', __( 'Invalid group', 'woocommerce-b2b' ), array( 'status' => 400 ) );
        }
        // Delete group
        $item = $this->get_item( array( 'id' => $request['id'] ) );
        $group_id = wp_delete_post( $request['id'], true );

        if ( ! $group_id ) {
            return new WP_Error( 'wcb2b_api_cannot_delete_group', __( 'This group cannot be deleted', 'woocommerce-b2b' ), array( 'status' => 400 ) );
        }
        return $item;
    }

    /**
     * Prepare data for respose
     */
    public function prepare_item_for_response( $group, $request ) {
        $group->wcb2b_group_discount_percentage = get_post_meta( $group->ID, 'wcb2b_group_discount', true );
        $group->wcb2b_group_min_purchase_amount = get_post_meta( $group->ID, 'wcb2b_group_min_purchase_amount', true );
        $group->wcb2b_group_packaging_fee = get_post_meta( $group->ID, 'wcb2b_group_packaging_fee', true );
        $group->wcb2b_group_terms_conditions = get_post_meta( $group->ID, 'wcb2b_group_terms_conditions', true );
        $group->wcb2b_group_disabled_shippings = get_post_meta( $group->ID, 'wcb2b_group_shippings', true );
        $group->wcb2b_group_disabled_gateways = get_post_meta( $group->ID, 'wcb2b_group_gateways', true );
        return $group;
    }

    /**
     * Prepare data for database
     */
    protected function prepare_item_for_database( $request ) {
        $meta_input = array();
        if ( isset( $request['wcb2b_group_discount'] ) ) {
            $meta_input['wcb2b_group_discount'] = $request['wcb2b_group_discount'];
        }
        if ( isset( $request['wcb2b_group_min_purchase_amount'] ) ) {
            $meta_input['wcb2b_group_min_purchase_amount'] = $request['wcb2b_group_min_purchase_amount'];
        }
        if ( isset( $request['wcb2b_group_packaging_fee'] ) ) {
            $meta_input['wcb2b_group_packaging_fee'] = $request['wcb2b_group_packaging_fee'];
        }
        if ( isset( $request['wcb2b_group_terms_conditions'] ) ) {
            $meta_input['wcb2b_group_terms_conditions'] = $request['wcb2b_group_terms_conditions'];
        }
        if ( isset( $request['wcb2b_group_shippings'] ) ) {
            $meta_input['wcb2b_group_shippings'] = $request['wcb2b_group_shippings'];
        }
        if ( isset( $request['wcb2b_group_gateways'] ) ) {
            $meta_input['wcb2b_group_gateways'] = $request['wcb2b_group_gateways'];
        }
        return array(
            'post_title' => $request['name'], 
            'post_type' => 'wcb2b_group',
            'post_status' => 'publish',
            'meta_input' => $meta_input
        );
    }

    /**
     * Check permissions
     */
    public function get_items_permissions_check( $request ) {
        if ( ! current_user_can( 'list_users' ) ) {
            return new WP_Error( 'wcb2b_api_cannot_read', __( 'Sorry, you cannot list groups', 'woocommerce-b2b' ), array( 'status' => rest_authorization_required_code() ) );
        }
        return true;
    }

    public function get_item_permissions_check( $request ) {
        if ( ! current_user_can( 'list_users' ) ) {
            return new WP_Error( 'wcb2b_api_cannot_read', __( 'Sorry, you cannot list groups', 'woocommerce-b2b' ), array( 'status' => rest_authorization_required_code() ) );
        }
        return true;
    }

    public function create_item_permissions_check( $request ) {
        if ( ! current_user_can( 'publish_products' ) ) {
            return new WP_Error( 'wcb2b_api_cannot_write', __( 'Sorry, you cannot add groups', 'woocommerce-b2b' ), array( 'status' => rest_authorization_required_code() ) );
        }
        return true;
    }

    public function update_item_permissions_check( $request ) {
        if ( ! current_user_can( 'edit_products' ) ) {
            return new WP_Error( 'wcb2b_api_cannot_edit', __( 'Sorry, you cannot edit groups', 'woocommerce-b2b' ), array( 'status' => rest_authorization_required_code() ) );
        }
        return true;
    }

    public function delete_item_permissions_check( $request ) {
        if ( ! current_user_can( 'delete_products' ) ) {
            return new WP_Error( 'wcb2b_api_cannot_delete', __( 'Sorry, you cannot delete groups', 'woocommerce-b2b' ), array( 'status' => rest_authorization_required_code() ) );
        }
        return true;
    }

}